package com.atlassian.bitbucket.pull;

import com.atlassian.bitbucket.commit.AbstractCommitsRequest;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Set;

/**
 * @since 6.0
 */
public class PullRequestCommitsRequest extends AbstractPullRequestRequest {

    private final int maxMessageLength;
    private final Set<String> propertyKeys;

    private PullRequestCommitsRequest(Builder builder) {
        super(builder);

        maxMessageLength = builder.maxMessageLength;
        propertyKeys = builder.propertyKeys.build();
    }

    public int getMaxMessageLength() {
        return maxMessageLength;
    }

    @Nonnull
    public Set<String> getPropertyKeys() {
        return propertyKeys;
    }

    public static class Builder extends AbstractBuilder<Builder> {

        private final ImmutableSet.Builder<String> propertyKeys = ImmutableSet.builder();

        private int maxMessageLength = AbstractCommitsRequest.UNLIMITED_MESSAGE_LENGTH;

        public Builder(@Nonnull PullRequest pullRequest) {
            super(pullRequest);
        }

        public Builder(int repositoryId, long pullRequestId) {
            super(repositoryId, pullRequestId);
        }

        @Nonnull
        public PullRequestCommitsRequest build() {
            return new PullRequestCommitsRequest(this);
        }

        @Nonnull
        public Builder maxMessageLength(int value) {
            maxMessageLength = value;

            return self();
        }

        @Nonnull
        public Builder propertyKey(@Nullable String value) {
            addIf(NOT_BLANK, propertyKeys, value);

            return self();
        }

        @Nonnull
        public Builder propertyKeys(@Nullable Iterable<String> values) {
            addIf(NOT_BLANK, propertyKeys, values);

            return self();
        }

        @Nonnull
        public Builder propertyKeys(@Nullable String value, @Nullable String... values) {
            addIf(NOT_BLANK, propertyKeys, value, values);

            return self();
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }
}
