package com.atlassian.bitbucket.label;

import com.atlassian.bitbucket.util.Page;
import com.atlassian.bitbucket.util.PageRequest;

import javax.annotation.Nonnull;
import java.util.Optional;

/**
 * Describes a service for interacting with labels.
 *
 * @since 5.12
 */
public interface LabelService {

    /**
     * Add a {@link Label} to a {@link Labelable}.
     *
     * @param request request containing the {@link Label} and {@link Labelable}
     * @return the {@link Label} that was added.
     */
    @Nonnull
    Label apply(@Nonnull ApplyLabelRequest request);

    /**
     * Creates a new label based on the given labelName.
     * <p>
     * The labelName has to be at least 3 characters and cannot exceed 50 characters, the labelName also has to be a
     * lowercase string consisting out of alpanumeric characters, "-" is also allowed.
     *
     * @param labelName the name for the newly created label.
     * @return the created {@link Label}
     */
    @Nonnull
    Label create(@Nonnull String labelName);

    /**
     * Retrieves a {@link Page} of {@link Label} for all the labels in the system.
     *
     * @param pageRequest the {@link PageRequest}
     * @return {@link Page} of {@link Label}
     */
    @Nonnull
    Page<Label> findAll(@Nonnull PageRequest pageRequest);

    /**
     * returns a {@link Page} of {@link Label labels} related to this {@link Labelable}
     *
     * @param labelable   the {@link Labelable} that will have it's related labels returned
     * @param pageRequest the {@link PageRequest}
     * @return {@link Page} of {@link Label}
     */
    @Nonnull
    Page<Label> findByLabelable(@Nonnull Labelable labelable, @Nonnull PageRequest pageRequest);

    /**
     * Returns an {@link Optional} containing a {@link Label} based on the name.
     *
     * @param labelName name of the {@link Label}
     * @return {@link Optional} containing a {@link Label}
     */
    @Nonnull
    Optional<Label> findByName(@Nonnull String labelName);

    /**
     * returns a {@link Page} of {@link Label labels} based on a prefix.
     *
     * @param labelPref   the prefix the {@link Label labels} have to start with
     * @param pageRequest the {@link PageRequest}
     * @return {@link Page} of  {@link Label}
     */
    @Nonnull
    Page<Label> findByPrefix(@Nonnull String labelPref, @Nonnull PageRequest pageRequest);

    /**
     * Returns a {@link Label} based on the name.
     *
     * @param labelName name of the {@link Label}
     * @return a {@link Label}
     */
    @Nonnull
    Label getByName(@Nonnull String labelName);

    /**
     * Returns {@link Page} of {@link Labelable labelables} for a given {@link LabelableSearchRequest}.
     *
     * @param searchRequest a {@link LabelableSearchRequest}
     * @param pageRequest the {@link PageRequest}
     * @return {@link Page} of {@link Labelable labelables}
     */
    @Nonnull
    Page<Labelable> searchLabelables(@Nonnull LabelableSearchRequest searchRequest, @Nonnull PageRequest pageRequest);

    /**
     * Removes a {@link Label} from a {@link Labelable}.
     *
     * @param request request containing the {@link Label} and {@link Labelable}
     */
    void remove(@Nonnull RemoveLabelRequest request);
}
