package com.atlassian.bitbucket.job;

import javax.annotation.Nonnull;

/**
 * @since 5.13
 */
public enum JobState {

    /**
     * The job is initialising in preparation for running
     */
    INITIALISING(100, false, false),
    /**
     * The job is ready to run and awaiting scheduling
     */
    READY(200, false, false),
    /**
     * The job is currently running
     */
    RUNNING(300, false, false),
    /**
     * The job is finalising its activities in preparation for stopping
     */
    FINALISING(400, false, false),
    /**
     * The job has run to completion and is now stopped. Errors or warning {@link JobMessage messages} may have been
     * logged determining whether the job failed or succeeded overall, depending on the type of job.
     */
    COMPLETED(500, false, true),
    /**
     * The job has run to completion and is now stopped. Errors or warning {@link JobMessage messages} may have been
     * logged determining whether the job failed or succeeded overall, depending on the type of job.
     */
    FAILED(600, true, true),
    /**
     * The job was canceled by the system or a user and is in the process of stopping.
     */
    CANCELING(650, true, false),
    /**
     * The job was canceled by the system or a user and has now stopped.
     */
    CANCELED(700, true, true),
    /**
     * The job ran for too long without an update and is now considered stopped
     */
    TIMED_OUT(800, true, true),
    /**
     * The job encountered an unrecoverable error and was forced to terminate abnormally
     */
    ABORTED(900, true, true);

    private final boolean failed;
    private final int id;
    private final boolean terminated;

    JobState(int id, boolean failed, boolean terminated) {
        this.id = id;
        this.failed = failed;
        this.terminated = terminated;
    }

    @Nonnull
    public static JobState fromId(int id) {
        for (JobState value : values()) {
            if (value.getId() == id) {
                return value;
            }
        }
        throw new IllegalArgumentException("No JobState is associated with ID [" + id + "]");
    }

    public int getId() {
        return id;
    }

    public boolean isFailed() {
        return failed;
    }

    public boolean isTerminated() {
        return terminated;
    }
}
