package com.atlassian.bitbucket.hook.script;

import javax.annotation.Nonnull;
import java.util.Date;
import java.util.Optional;

/**
 * Describes a hook script.
 * <p>
 * A script represents some form of executable, such as a shell script or binary, which can be run in response to
 * repository changes. Scripts are managed separately from the repositories where they're enabled to simplify the
 * process of administering them. For example, if a given script is enabled in 2,000 different repositories, it can
 * be updated in, or deleted from, all of them simultaneously with a single operation.
 *
 * @since 6.2
 */
public interface HookScript extends MinimalHookScript {

    /**
     * @return the date the script was created
     */
    @Nonnull
    Date getCreatedDate();

    /**
     * Retrieves the script's description. Descriptions are limited to 255 characters but are otherwise free-form.
     *
     * @return the script's description
     */
    @Nonnull
    Optional<String> getDescription();

    /**
     * Retrieves the script's name. Names are limited to 255 characters but are otherwise free-form.
     * <p>
     * When a {@link HookScriptType#PRE pre}-style hook script rejects an update, the script's name is included
     * in the {@link com.atlassian.bitbucket.hook.repository.RepositoryHookVeto veto} added to the
     * {@link com.atlassian.bitbucket.hook.repository.RepositoryHookResult hook result}. Like a filename, hook
     * script names should be short, but descriptive. (However, unlike a filename, filesystem limitations <i>do
     * not apply</i> to hook script names.)
     *
     * @return the script's name
     */
    @Nonnull
    String getName();

    /**
     * Retrieves the plugin key for the app which created the script. Plugin keys are limited to 255 characters.
     * <p>
     * When a script is created, the app which creates it must provide its plugin key. This allows scripts to be
     * associated with the app that created them, for administrative purposes.
     * <p>
     * <b>Note</b>: App developers are <i>strongly discouraged</i> from changing their plugin keys. While the key
     * associated with a hook script <i>can</i> be updated later, this is only one of many references the overall
     * system may have to an app's plugin key.
     *
     * @return the plugin key for the app which created the script
     */
    @Nonnull
    String getPluginKey();

    /**
     * @return the size of the script's contents
     */
    int getSize();

    /**
     * Retrieves the script's type.
     * <p>
     * Hook scripts can be written to be invoked as {@code PreRepositoryHook}s or {@code PostRepositoryHook}s.
     * The type must be specified when the script is created, and cannot be updated after.
     *
     * @return the script's type, indicating whether it will be invoked before or after changes are accepted
     */
    @Nonnull
    HookScriptType getType();

    /**
     * @return the date the script was last updated
     */
    @Nonnull
    Date getUpdatedDate();
}
