package com.atlassian.bitbucket.hook.repository;

import com.atlassian.bitbucket.ServiceException;
import com.atlassian.bitbucket.i18n.KeyedMessage;

import javax.annotation.Nonnull;
import java.util.List;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Objects.requireNonNull;

/**
 * Exception that can be thrown when one or more {@code repository-hook} modules reject a proposed ref changes. Note
 * that {@link RepositoryHookService#preUpdate(RepositoryHookRequest)} does not throw this exception. However, code
 * that invokes the hooks may throw this exception after one or more hooks have rejected the exception.
 *
 * @see RepositoryHookService#preUpdate(RepositoryHookRequest)
 * @since 5.0
 */
public class RepositoryHookVetoedException extends ServiceException {

    private final RepositoryHookRequest request;
    private final List<RepositoryHookVeto> vetoes;

    public RepositoryHookVetoedException(@Nonnull KeyedMessage message, @Nonnull RepositoryHookRequest request,
                                         @Nonnull List<RepositoryHookVeto> vetoes) {
        super(message);

        checkArgument(!requireNonNull(vetoes, "vetoes").isEmpty(), "At least one veto must be provided");

        this.request = requireNonNull(request, "request");
        this.vetoes = vetoes;
    }

    /**
     * @param <T> type of request
     * @return the request that was rejected
     */
    @Nonnull
    public <T extends RepositoryHookRequest> T getRequest() {
        //noinspection unchecked
        return (T) request;
    }

    /**
     * @return the reasons for rejecting the request
     */
    @Nonnull
    public List<RepositoryHookVeto> getVetoes() {
        return vetoes;
    }
}
