package com.atlassian.bitbucket.event.repository;

import com.atlassian.analytics.api.annotations.EventName;
import com.atlassian.audit.entity.CoverageLevel;
import com.atlassian.bitbucket.event.annotation.TransactionAware;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.event.api.AsynchronousPreferred;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;

import static java.util.Objects.requireNonNull;

/**
 * Event that is raised when a repository is deleted
 * <p>
 * This event is internally audited with {@link CoverageLevel#BASE} level.
 */
@AsynchronousPreferred
@TransactionAware
@EventName("stash.repository.deleted")
public class RepositoryDeletedEvent extends RepositoryEvent {

    private final Iterable<Integer> forkIds;

    /**
     * @param source        the object on which the event initially occurred
     * @param repository    repository that was deleted
     * @param forkIds       IDs of forks
     * @since 5.6
     */
    public RepositoryDeletedEvent(@Nonnull Object source, @Nonnull Repository repository,
                                  @Nonnull Iterable<Integer> forkIds) {
        super(source, repository);

        this.forkIds = ImmutableSet.copyOf(requireNonNull(forkIds, "forkIds"));
    }

    /**
     * @return the IDs of all repositories that are forks of the repository being deleted
     * @since 5.6
     */
    @Nonnull
    public Iterable<Integer> getForkIds() {
        return forkIds;
    }
}
