package com.atlassian.bitbucket.event.project;

import com.atlassian.audit.entity.CoverageLevel;
import com.atlassian.bitbucket.event.CancelableEvent;
import com.atlassian.bitbucket.i18n.KeyedMessage;
import com.atlassian.bitbucket.project.Project;
import com.atlassian.bitbucket.project.ProjectService;
import com.atlassian.bitbucket.project.ProjectUpdateRequest;
import com.atlassian.bitbucket.util.CancelState;

import javax.annotation.Nonnull;

import static java.util.Objects.requireNonNull;

/**
 * Raised just before a {@link Project project} is updated. This event is synchronous, allowing listeners to
 * perform operations in the same database transaction where the project will be updated.
 * <p>
 * When this event is raised, the project's persisted state has <i>not</i> been updated, so {@link #getProject()}
 * matches the {@link #getOldValue() old values}.
 * <p>
 * This event is {@link CancelableEvent cancelable}. A listener may prevent the project from being updated by
 * {@link #cancel(KeyedMessage) canceling} this event. Throwing an exception <i>will not</i> prevent the project
 * from being updated; the exception will be logged and ignored.
 * <p>
 * This event is internally audited at the {@link CoverageLevel#BASE} level.
 *
 * @see ProjectService#update(ProjectUpdateRequest)
 */
public class ProjectModificationRequestedEvent
        extends AbstractProjectModificationEvent
        implements CancelableEvent {

    private final CancelState cancelState;

    public ProjectModificationRequestedEvent(@Nonnull Object source, @Nonnull Project oldValue,
                                             @Nonnull Project newValue, @Nonnull CancelState cancelState) {
        super(source, oldValue, oldValue, newValue);

        this.cancelState = requireNonNull(cancelState, "cancelState");
    }

    /**
     * Cancels project modification, providing a message explaining why.
     *
     * @param message a descriptive message explaining why the operation has been canceled
     */
    @Override
    public void cancel(@Nonnull KeyedMessage message) {
        cancelState.cancel(message);
    }

    /**
     * Retrieves a flag indicating whether project modification has already been canceled by another listener.
     *
     * @return {@code true} if another listener has already canceled project modification; otherwise, {@code false}
     */
    @Override
    public boolean isCanceled() {
        return cancelState.isCanceled();
    }
}
