package com.atlassian.bitbucket.event.audit;

import com.atlassian.audit.api.AuditConsumer;
import com.atlassian.bitbucket.audit.AuditEntry;
import com.atlassian.bitbucket.audit.AuditEntryConverter;
import com.atlassian.bitbucket.audit.Priority;
import com.atlassian.bitbucket.event.ApplicationEvent;
import com.atlassian.bitbucket.event.annotation.Audited;
import com.atlassian.event.api.AsynchronousPreferred;

import javax.annotation.Nonnull;
import java.util.Collections;
import java.util.Set;

import static java.util.Objects.requireNonNull;

/**
 * Represents an {@link AuditEvent} derived from an {@link Audited} annotated event or an internal auditable action.
 * <p>
 * This class can be used in two different ways. The primary (intended) use case is for <strong>consuming</strong> all
 * auditable actions, thus offering a consumer API for these actions. The system publishes this event internally for every
 * auditable action, and a listener can be created to listen to {@link AuditEvent} and process it (e.g. add it to a file
 * log, persist it in the database, etc.).
 * <p>
 * The secondary use case is a low level API for <strong>providing</strong> auditable actions to the system, thus offering
 * a provider API for these actions. An {@link AuditEvent} can be published directly to add an auditable action, instead
 * of using the {@link Audited} annotation and an {@link AuditEntryConverter}. However this is not the preferred method
 * of providing an auditable action (as it skips over system processing applied to every {@link Audited} event as it is
 * transformed into an {@link AuditEvent}), and as such it is only recommended to be used if {@link Audited} is not
 * suitable in a particular situation.
 * <p>
 * The {@link AuditEvent} class will be retained until 8.0 to continue supporting provider API
 * for auditable actions. However, the consumer API would be removed in 7.0, i.e., the system would stop publishing
 * {@link AuditEvent}s and so, listeners of this event would no longer receive it and be notified of auditable actions.
 *
 * @deprecated the system internally publishing {@link AuditEvent} for consumers to listen to will be removed in 7.0.
 *             Use {@link AuditConsumer} instead to consume auditable actions.
 */
@AsynchronousPreferred
@Deprecated
public class AuditEvent extends ApplicationEvent {

    private final AuditEntry entry;
    private final Set<String> channels;
    private final Priority priority;

    public AuditEvent(@Nonnull Object source, @Nonnull AuditEntry entry, @Nonnull Set<String> channels, Priority priority) {
        super(requireNonNull(source, "source"));
        this.priority = priority;
        this.entry = requireNonNull(entry, "entry");
        this.channels = Collections.unmodifiableSet(requireNonNull(channels, "channels"));
    }

    /**
     * @return the details of the original event in a standard audit format
     * */
    @Nonnull
    public AuditEntry getEntry() {
        return entry;
    }

    /**
      * @return the channels which the original annotated event indicated would be interested in auditing the original event
     */
    @Nonnull
    public Set<String> getChannels() {
        return channels;
    }

    /**
     * @return the priority of the event
     * */
    @Nonnull
    public Priority getPriority() {
        return priority;
    }
}
