package com.atlassian.bitbucket.comment;

import com.atlassian.bitbucket.pull.PullRequest;

import javax.annotation.Nonnull;
import java.util.stream.Stream;

/**
 * Enumerates the different types of diffs a {@link CommentThread thread} can be anchored to.
 *
 * @since 5.0
 */
public enum CommentThreadDiffAnchorType {

    /**
     * A commit diff shows changes introduced by a single commit.
     */
    COMMIT(1),
    /**
     * An effective diff shows all the changes that would be introduced to a {@link PullRequest pull request}
     * target branch.
     */
    EFFECTIVE(0),
    /**
     * A range diff displays the changes introduced by a number of commits.
     */
    RANGE(2);

    private final int id;

    CommentThreadDiffAnchorType(int id) {
        this.id = id;
    }

    @Nonnull
    public static CommentThreadDiffAnchorType fromId(int id) {
        return Stream.of(values())
                .filter(diff -> diff.getId() == id)
                .findFirst()
                .orElseThrow(() ->
                        new IllegalArgumentException("No CommentThreadDiffAnchorType is associated with ID [" + id + "]"));
    }

    public int getId() {
        return id;
    }
}
