package com.atlassian.bitbucket.build.status;

import com.atlassian.bitbucket.build.BuildState;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.util.BuilderSupport;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static java.util.Objects.requireNonNull;

/**
 * @since 7.6
 */
public final class RepositoryBuildStatusSetRequest {

    private final String buildNumber;
    private final String commitId;
    private final String description;
    private final Long duration;
    private final String key;
    private final String name;
    private final String parent;
    private final String ref;
    private final Repository repository;
    private final BuildState state;
    private final TestResults testResults;
    private final String url;

    private RepositoryBuildStatusSetRequest(Builder builder) {
        buildNumber = builder.buildNumber;
        commitId = builder.commitId;
        description = builder.description;
        duration = builder.duration;
        key = builder.key;
        name = builder.name;
        parent = builder.parent;
        ref = builder.ref;
        repository = builder.repository;
        state = builder.state;
        testResults = builder.testResults;
        url = builder.url;
    }

    /**
     * @return an identifier for the specific run that resulted in this build status
     */
    @Nullable
    public String getBuildNumber() {
        return buildNumber;
    }

    /**
     * @return the commit hash that the build was run against
     */
    @Nonnull
    public String getCommitId() {
        return commitId;
    }

    /**
     * @return a short description of the build status
     */
    @Nullable
    public String getDescription() {
        return description;
    }

    /**
     * @return the length of time taken to run the build, in seconds
     */
    @Nullable
    public Long getDuration() {
        return duration;
    }

    /**
     * @return the key for the build that this status is associated with
     */
    @Nonnull
    public String getKey() {
        return key;
    }

    /**
     * @return the name of the build
     */
    @Nullable
    public String getName() {
        return name;
    }

    /**
     * @return the identifier for the parent of the plan or job that produced this build status
     */
    @Nullable
    public String getParent() {
        return parent;
    }

    /**
     * @return the fully qualified ref (e.g. refs/heads/master) that this build was run against, or {@code null} if it is
     *         not known or not run on a ref
     */
    @Nullable
    public String getRef() {
        return ref;
    }

    /**
     * @return the repository that this build status was built against
     */
    @Nonnull
    public Repository getRepository() {
        return repository;
    }

    /**
     * @return the resulting {@link BuildState} of the build
     */
    @Nonnull
    public BuildState getState() {
        return state;
    }

    /**
     * @return the {@link TestResults test results} for this build
     */
    @Nullable
    public TestResults getTestResults() {
        return testResults;
    }

    /**
     * @return the URL to navigate to this build status in the CI tool
     */
    @Nonnull
    public String getUrl() {
        return url;
    }

    public static class Builder extends BuilderSupport {

        private final String commitId;
        private final String key;
        private final Repository repository;

        private String buildNumber;
        private String description;
        private Long duration;
        private String name;
        private String parent;
        private String ref;
        private BuildState state;
        private TestResults testResults;
        private String url;

        public Builder(@Nonnull Repository repository, @Nonnull String commitId, @Nonnull String key) {
            this.repository = requireNonNull(repository, "repository");
            this.commitId = commitId;
            this.key = requireNonBlank(key, "key");
        }

        @Nonnull
        public RepositoryBuildStatusSetRequest build() {
            checkNotBlank(key, "key");
            requireNonNull(state, "state");
            checkNotBlank(url, "url");

            return new RepositoryBuildStatusSetRequest(this);
        }

        @Nonnull
        public Builder buildNumber(@Nullable String value) {
            buildNumber = StringUtils.trimToNull(value);
            if (StringUtils.length(ref) > 255) {
                throw new IllegalArgumentException("The buildNumber can't be longer than 1024");
            }

            return this;
        }

        @Nonnull
        public Builder description(@Nullable String value) {
            description = StringUtils.trimToNull(StringUtils.abbreviate(value, 255));

            return this;
        }

        @Nonnull
        public Builder duration(@Nullable Long value) {
            duration = value;

            return this;
        }

        @Nonnull
        public Builder name(@Nullable String value) {
            name = StringUtils.trimToNull(value);

            return this;
        }

        @Nonnull
        public Builder parent(@Nullable String value) {
            parent = StringUtils.trimToNull(value);
            if (StringUtils.length(ref) > 1024) {
                throw new IllegalArgumentException("The parent can't be longer than 1024");
            }

            return this;
        }

        @Nonnull
        public Builder ref(@Nullable String value) {
            ref = StringUtils.trimToNull(value);
            if (StringUtils.length(ref) > 1024) {
                throw new IllegalArgumentException("The ref can't be longer than 1024");
            }

            return this;
        }

        @Nonnull
        public Builder state(@Nonnull BuildState value) {
            state = requireNonNull(value, "state");

            return this;
        }

        @Nonnull
        public Builder testResults(@Nonnull TestResults testResults) {
            this.testResults = testResults;

            return this;
        }

        @Nonnull
        public Builder url(@Nonnull String value) {
            url = checkNotBlank(value, "url");

            return this;
        }
    }
}
