package com.atlassian.bitbucket.build.status;

import com.atlassian.bitbucket.build.BuildState;

import com.atlassian.bitbucket.commit.Commit;
import com.atlassian.bitbucket.repository.Repository;

import javax.annotation.Nonnull;
import java.util.Date;
import java.util.Optional;

/**
 * A build result associated with a commit in a repository.
 *
 * @since 7.6
 */
public interface RepositoryBuildStatus {

    /**
     * An identifier for the specific run that resulted in this build status.
     *
     * For example, in Bamboo the URL to a build might be https://my-bamboo-url.com/browse/PROJ-PLAN123-5 for the 5th
     * run of the PLAN123 branch plan of the parent plan PLAN in the project PROJ. In this case the build number would
     * be '5'.
     *
     * @return the build number if one exists, or {@link Optional#empty()} otherwise
     */
    @Nonnull
    Optional<String> getBuildNumber();

    /**
     * @return the build server that posted this build status, if known, {@link Optional#empty()} otherwise
     * @since 7.8
     */
    @Nonnull
    Optional<BuildServer> getBuildServer();

    /**
     * Returns the commit this build status is attached to. This will not be present in build statuses created using the
     * build status rest API (instead of the repository build status rest API).
     *
     * @return the commit associated with the build, if there is one, {@link Optional#empty()} otherwise
     * @since 7.8
     */
    @Nonnull
    Optional<Commit> getCommit();

    /**
     * @return the commit ID that the build ran against
     */
    @Nonnull
    String getCommitId();

    /**
     * @return all external links associated with the build status
     * @since 7.8
     */
    @Nonnull
    BuildStatusLinks getBuildStatusLinks();

    /**
     * The date tha the build status was first created.
     *
     * If the build status is subsequently updated (e.g. to change the state from INPROGRESS to SUCCESSFUL, the created
     * date will remain the same but the {@link #getUpdatedDate()} will change.
     *
     * @return the date that the build status was first created
     */
    @Nonnull
    Date getCreatedDate();

    /**
     * A human readable description of the build that was run.
     *
     * @return the description of the build that this build status represents, or {@link Optional#empty()} if none exists
     */
    @Nonnull
    Optional<String> getDescription();

    /**
     * @return the duration of the build in ms, or {@code null} if the duration is not provided
     * @since 7.8
     */
    @Nonnull
    Optional<Long> getDuration();

    /**
     * The key representing the build.
     *
     * The key should be a unique identifier when combined with the repository and the commit ID.
     * For example, in Bamboo the URL to a build might be https://my-bamboo-url.com/browse/PROJ-PLAN123-5 for the 5th
     * run of the PLAN123 branch plan of the parent plan PLAN in the project PROJ. In this case, the key would be
     * PROJ-PLAN123
     *
     * @return the build key
     */
    @Nonnull
    String getKey();

    /**
     * A human readable name of the build that was run.
     *
     * @return the name of the build, or {@link Optional#empty()} is none exists
     */
    @Nonnull
    Optional<String> getName();

    /**
     * The key of the parent plan for this build.
     *
     * For example, in Bamboo the URL to a build might be https://my-bamboo-url.com/browse/PROJ-PLAN123-5 for the 5th
     * run of the PLAN123 branch plan of the parent plan PLAN in the project PROJ. In this case, the parent key would
     * be PROJ-PLAN (where as the key would be PROJ-PLAN123).
     *
     * @return the identifier for the parent of the plan or job that produced this build status, if there is one,
     *         {@link Optional#empty()} otherwise
     */
    @Nonnull
    Optional<String> getParent();

    /**
     * The fully qualified ref that the build is associated with.
     *
     * @return the branch or tag that this build result is associated with, or {@link Optional#empty()} if the CI tool
     *         did not provide one
     */
    @Nonnull
    Optional<String> getRef();

    /**
     * The repository that the build is associated with.
     *
     * @return the repository for this build status
     */
    @Nonnull
    Repository getRepository();

    /**
     * The outcome of the build ({@link BuildState#SUCCESSFUL} or {@link BuildState#FAILED} or
     * {@link BuildState#INPROGRESS} if the build hasn't completed yet.
     *
     * @return the {@link BuildState}
     */
    @Nonnull
    BuildState getState();

    /**
     * A high level summary of which tests passed and failed.
     *
     * @return the {@link TestResults} for the build, if there are any, {@link Optional#empty()} otherwise
     */
    @Nonnull
    Optional<TestResults> getTestResults();

    /**
     * The date that the build status for this repository, commitId and key was last updated.
     *
     * @return the date this build status was last updated
     */
    @Nonnull
    Date getUpdatedDate();

    /**
     * A url to this build in the CI tool that ran it, so that users are able to navigate there for more information.
     *
     * @return a link to the build
     */
    @Nonnull
    String getUrl();
}
