package com.atlassian.bitbucket.audit;

import com.atlassian.audit.entity.AuditEvent;
import com.atlassian.bitbucket.event.annotation.Audited;
import com.atlassian.bitbucket.project.Project;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.user.ApplicationUser;
import com.atlassian.bitbucket.util.BuilderSupport;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Date;

import static java.util.Objects.requireNonNull;

/**
 * A builder for constructing {@link AuditEntry audit entries}.
 * <p>
 * Supplying an {@link #action(String) action} is required. If no action is supplied to the builder, calling
 * {@link #build()} will throw a {@link NullPointerException}.
 *
 * @see AuditEntry
 * @see AuditEntryConverter
 * @see Audited
 * @deprecated in 7.0 for removal in 8.0. {@link AuditEvent.Builder AuditEvent.Builder} should be used
 * to build {@link AuditEvent}s instead.
 */
@Deprecated
public class AuditEntryBuilder extends BuilderSupport {

    private String action;
    private String details;
    private Project project;
    private Repository repository;
    private String sourceIpAddress;
    private String target;
    private long timestamp;
    private ApplicationUser user;

    @Nonnull
    public AuditEntryBuilder action(@Nonnull Class<?> value) {
        return action(requireNonNull(value, "action").getSimpleName());
    }

    @Nonnull
    public AuditEntryBuilder action(@Nonnull String value) {
        requireNonNull(value, "action");
        if (value.startsWith("Analytics") && !"Analytics".equals(value)) {
            value = value.substring(9);
        }
        action = checkNotBlank(value, "action");

        return this;
    }

    /**
     * Return the resultant {@link AuditEntry}.
     * <p>
     * Supplying an {@link #action(String) action} is required. If no action is supplied to the builder, calling
     * {@link #build()} will throw a {@link NullPointerException}.
     *
     * @return the resultant {@link AuditEntry}
     */
    @Nonnull
    public AuditEntry build() {
        return new SimpleAuditEntry(this);
    }

    @Nonnull
    public AuditEntryBuilder details(@Nullable String value) {
        details = value;

        return this;
    }

    @Nonnull
    public AuditEntryBuilder project(@Nullable Project value) {
        project = value;

        return this;
    }

    @Nonnull
    public AuditEntryBuilder repository(@Nullable Repository value) {
        repository = value;
        if (value != null) {
            project = value.getProject();
        }

        return this;
    }

    @Nonnull
    public AuditEntryBuilder sourceIpAddress(@Nullable String value) {
        sourceIpAddress = value;

        return this;
    }

    @Nonnull
    public AuditEntryBuilder target(@Nullable String value) {
        target = value;

        return this;
    }

    @Nonnull
    public AuditEntryBuilder timestamp(@Nonnull Date value) {
        timestamp = requireNonNull(value, "timestamp").getTime();

        return this;
    }

    @Nonnull
    public AuditEntryBuilder timestamp(long value) {
        timestamp = value;

        return this;
    }

    @Nonnull
    public AuditEntryBuilder user(@Nullable ApplicationUser value) {
        user = value;

        return this;
    }

    /**
     * A simple, immutable implementation of {@link AuditEntry}.
     */
    private static class SimpleAuditEntry implements AuditEntry {

        private final Project project;
        private final Repository repository;
        private final ApplicationUser user;
        private final String details;
        private final long timestamp;
        private final String action;
        private final String sourceIpAddress;
        private final String target;

        private SimpleAuditEntry(AuditEntryBuilder builder) {
            action = requireNonNull(builder.action, "action");
            details = builder.details;
            project = builder.project;
            repository = builder.repository;
            sourceIpAddress = builder.sourceIpAddress;
            target = builder.target;
            timestamp = builder.timestamp;
            user = builder.user;
        }

        /**
         * @return the type of event occuring
         */
        @Nonnull
        @Override
        public String getAction() {
            return action;
        }

        /**
         * @return relevant details of the audited event
         */
        @Override
        public String getDetails() {
            return details;
        }

        /**
         * @return the affected project
         */
        @Override
        public Project getProject() {
            return project;
        }

        /**
         * @return the affected repository
         */
        @Override
        public Repository getRepository() {
            return repository;
        }

        /**
         * @return time the audited event was received
         */
        @Nonnull
        @Override
        public Date getTimestamp() {
            return new Date(timestamp);
        }

        /**
         * @return the source IP address of the user causing the audit event
         */
        @Override
        public String getSourceIpAddress() {
            return sourceIpAddress;
        }

        /**
         * @return details of the affected component (e.g. a plugin key or project slug)
         */
        @Nullable
        @Override
        public String getTarget() {
            return target;
        }

        /**
         * @return the user causing the audited event
         */
        @Override
        public ApplicationUser getUser() {
            return user;
        }
    }
}
