package com.atlassian.bitbucket.validation;

import com.atlassian.bitbucket.validation.annotation.ConditionalNotEmpty;
import org.apache.commons.lang3.StringUtils;

import javax.validation.ConstraintValidator;
import javax.validation.ConstraintValidatorContext;
import java.lang.reflect.Field;

public class ConditionalNotEmptyValidator implements ConstraintValidator<ConditionalNotEmpty, Object> {

    private String[] fields;
    private String message;
    private String targetField;

    @Override
    public void initialize(ConditionalNotEmpty constraint) {
        fields = constraint.fields();
        targetField = constraint.targetField();
        message = constraint.message();
    }

    @Override
    public boolean isValid(Object object, ConstraintValidatorContext context) {
        boolean valid = true;
        if (isBlank(object, targetField) && oneOrMoreFieldsNotEmpty(object, fields)) {
            context.disableDefaultConstraintViolation();
            context.buildConstraintViolationWithTemplate(message).addNode(targetField).addConstraintViolation();
            valid = false;
        }
        return valid;
    }

    private boolean oneOrMoreFieldsNotEmpty(Object value, String[] fields) {
        for (String fieldName : fields) {
            if (!isBlank(value, fieldName)) {
                return true;
            }
        }
        return false;
    }

    private boolean isBlank(final Object value, final String fieldName) {
        try {
            final Field field = findField(value, fieldName);

            if (!field.isAccessible()) {
                field.setAccessible(true);
            }

            final Object fieldValue = field.get(value);

            return fieldValue == null || StringUtils.isBlank(fieldValue.toString());
        } catch (NoSuchFieldException e) {
            throw new IllegalStateException("Invalid field: " + fieldName);
        } catch (IllegalAccessException e) {
            throw new IllegalStateException("Inaccessible field: " + fieldName);
        }
    }

    // based on Spring's ReflectionUtils.findFields()
    private Field findField(Object value, String name) throws NoSuchFieldException {
        Class<?> clazz = value.getClass();
        while (!Object.class.equals(clazz) && clazz != null) {
            final Field[] fields = clazz.getDeclaredFields();
            for (Field field : fields) {
                if (name.equals(field.getName())) {
                    return field;
                }
            }
            clazz = clazz.getSuperclass();
        }

        throw new NoSuchFieldException(name);
    }


}
