package com.atlassian.bitbucket.util;


import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;

import javax.annotation.Nonnull;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.function.Function;

/**
 * An instance of {@link Page} that supports pagination across an underlying paged collection.
 * @param <T>
 */
public class FilteredPageImpl<T> implements Page<T> {

    private final PageRequest pageRequest;
    private final int size;
    private final SortedMap<Integer, T> valueMap;
    private final boolean lastPage;
    private final int nextOrdinal;

    public FilteredPageImpl(@Nonnull PageRequest pageRequest, int size, @Nonnull SortedMap<Integer, T> valueMap, boolean lastPage) {
        this.pageRequest = pageRequest;
        this.size = size;
        this.lastPage = lastPage;
        
        if (valueMap.size() > size) {
            nextOrdinal = Iterables.get(valueMap.keySet(), size);
            this.valueMap = valueMap.headMap(nextOrdinal);
        } else {
            this.valueMap = valueMap;
            if (!valueMap.isEmpty()) {
                nextOrdinal = valueMap.lastKey() + 1;
            } else {
                nextOrdinal = -1;
            }
        }
    }

    @Override
    public int getStart() {
        return pageRequest.getStart();
    }

    @Override
    public int getSize() {
        return size;
    }

    @Override
    public int getLimit() {
        return pageRequest.getLimit();
    }

    @Override
    public boolean getIsLastPage() {
        return lastPage;
    }

    @Nonnull
    @Override
    public Iterable<T> getValues() {
        return valueMap.values();
    }

    @Override
    public PageRequest getNextPageRequest() {
        if (lastPage) {
            return null;
        } else {
            return new PageRequestImpl(nextOrdinal, pageRequest.getLimit());
        }
    }

    @Override
    public SortedMap<Integer, T> getOrdinalIndexedValues() {
        return valueMap;
    }

    @Nonnull
    @Override
    public <E> FilteredPageImpl<E> transform(@Nonnull Function<? super T, ? extends E> transformFunction) {
        SortedMap<Integer, E> transformedValueMap = new TreeMap<>(Maps.transformValues(valueMap, transformFunction::apply));

        return new FilteredPageImpl<>(pageRequest, size, transformedValueMap, lastPage);
    }
}

