package com.atlassian.bitbucket.task;

import com.atlassian.bitbucket.util.BuilderSupport;

import javax.annotation.Nonnull;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * Allows the creation of a {@link Task task}.
 *
 * @see TaskService#create(TaskCreateRequest)
 */
public class TaskCreateRequest {

    private final long anchorId;
    private final TaskAnchorType anchorType;
    private final String text;

    private TaskCreateRequest(Builder builder) {
        checkArgument(builder.anchorId > 0, "anchor id");
        checkArgument(isNotBlank(builder.text), "the text must be non-blank");
        this.anchorType = checkNotNull(builder.anchorType, "anchor type");
        this.anchorId = builder.anchorId;
        this.text = builder.text;
    }

    public long getAnchorId() {
        return anchorId;
    }

    @Nonnull
    public TaskAnchorType getAnchorType() {
        return anchorType;
    }

    @Nonnull
    public String getText() {
        return text;
    }

    public static class Builder extends BuilderSupport {

        private long anchorId = -1;
        private TaskAnchorType anchorType;
        private String text;

        @Nonnull
        public Builder anchorId(long value) {
            checkArgument(value > 0, "anchor id");
            this.anchorId = value;
            return this;
        }

        @Nonnull
        public Builder anchorType(@Nonnull TaskAnchorType value) {
            this.anchorType = checkNotNull(value, "anchor");
            return this;
        }

        @Nonnull
        public Builder text(@Nonnull String value) {
            this.text = checkNotNull(value, "text");
            return this;
        }

        @Nonnull
        public TaskCreateRequest build() {
            return new TaskCreateRequest(this);
        }
    }
}
