package com.atlassian.bitbucket.scm.mirror;

import com.atlassian.bitbucket.scm.AbstractCommandParameters;

import javax.annotation.Nonnull;
import java.io.File;
import java.nio.file.Path;
import java.util.Optional;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;

/**
 * @since 4.1
 */
public class MirrorSyncCommandParameters extends AbstractCommandParameters {

    private final boolean includePrivateRefs;
    private final String password;
    private final File privateKeyFile;
    private final String remoteUrl;
    private final String username;

    private MirrorSyncCommandParameters(Builder builder) {
        this.includePrivateRefs = builder.includePrivateRefs;
        this.password = builder.password;
        this.privateKeyFile = builder.privateKeyFile;
        this.remoteUrl = builder.remoteUrl;
        this.username = builder.username;
    }

    /**
     * @return the password to provide for authentication
     * @since 4.6
     */
    @Nonnull
    public Optional<String> getPassword() {
        return ofNullable(password);
    }

    /**
     * @return the username to provide for authentication
     * @since 4.6
     */
    @Nonnull
    public Optional<String> getUsername() {
        return ofNullable(username);
    }

    /**
     * @return the SSH private key to use for authentication
     */
    @Nonnull
    public Optional<File> getPrivateKey() {
        return ofNullable(privateKeyFile);
    }

    /**
     * @return the URL to fetch from
     */
    @Nonnull
    public String getRemoteUrl() {
        return remoteUrl;
    }

    /**
     * @return true if private refs (outside refs/heads and refs/tags) should be included in the fetch
     * @since 5.1
     */
    public boolean isIncludePrivateRefs() {
        return includePrivateRefs;
    }

    public static class Builder {

        private final String remoteUrl;

        private boolean includePrivateRefs;
        private String password;
        private File privateKeyFile;
        private String username;

        /**
         * @param remoteUrl the URL to fetch from
         */
        public Builder(@Nonnull String remoteUrl) {
            this.remoteUrl = requireNonBlank(remoteUrl, "remoteUrl");
            includePrivateRefs = true;
        }

        @Nonnull
        public MirrorSyncCommandParameters build() {
            return new MirrorSyncCommandParameters(this);
        }

        /**
         * @param value whether to include private refs (outside refs/heads and refs/tags) in the fetch. The default
         *              value is true so all refs will be fetched.
         * @return the builder
         * @since 5.1
         */
        @Nonnull
        public Builder includePrivateRefs(boolean value) {
            includePrivateRefs = value;
            return this;
        }

        /**
         * @param value the password to use
         * @return the builder
         * @since 4.6
         */
        @Nonnull
        public Builder password(@Nonnull String value) {
            password = requireNonNull(value, "password");
            return this;
        }

        /**
         * @param value the SSH private key to use for authentication
         * @return the builder
         */
        @Nonnull
        public Builder privateKey(@Nonnull File value) {
            privateKeyFile = requireNonNull(value, "privateKeyFile");
            return this;
        }

        /**
         * @param value the SSH private key to use for authentication
         * @return the builder
         * @since 5.12
         */
        @Nonnull
        public Builder privateKey(@Nonnull Path value) {
            return privateKey(requireNonNull(value, "privateKeyFile").toFile());
        }

        /**
         * @param value the username to use
         * @return the builder
         * @since 4.6
         */
        @Nonnull
        public Builder username(@Nonnull String value) {
            username = requireNonNull(value, "value");
            return this;
        }
    }
}
