package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.repository.Repository;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

public class MergeCommandParameters extends AbstractMergeCommandParameters {

    private final String fromBranch;
    private final String fromCommitId;
    private final Repository fromRepository;
    private final String toBranch;

    protected MergeCommandParameters(AbstractMergeBuilder builder) {
        super(builder);

        fromBranch = builder.fromBranch;
        fromCommitId = builder.fromCommitId;
        fromRepository = builder.fromRepository;
        toBranch = builder.toBranch;
    }

    @Nonnull
    public String getFromBranch() {
        return fromBranch;
    }

    /**
     * Retrieves the specific commit that should be merged into the {@link #getToBranch() target branch}.
     * <p>
     * If this value is not provided, the <i>current tip</i> of the {@link #getFromBranch() source branch} is
     * merged into the target. This should be used with caution, as it may result in the system performing a
     * different merge than intended if the source branch changes between when the merge is requested and when
     * it is performed.
     *
     * @return the specific commit to merge in, or {@code null} to merge the current tip of the
     *         {@link #getFromBranch() branch}
     */
    @Nullable
    public String getFromCommitId() {
        return fromCommitId;
    }

    /**
     * Retrieves the {@link Repository repository} containing the from {@link #getFromBranch()} and
     * {@link #getFromCommitId() commit}.
     *
     * @return the from repository, or {@code null} if the from and to branches are in the same repository
     */
    @Nullable
    public Repository getFromRepository() {
        return fromRepository;
    }

    @Nonnull
    public String getToBranch() {
        return toBranch;
    }

    /**
     * Retrieves a flag indicating whether these parameters include a {@link #getFromRepository() from repository},
     * indicating the merge is cross-repository, where the to repository is provided when requesting the merge command
     * rather than in these parameters.
     *
     * @return {@code true} if a {@link #getFromRepository() from repository} is set; otherwise, {@code false}
     */
    public boolean hasFromRepository() {
        return fromRepository != null;
    }

    /**
     * @param <B> the concrete builder class
     * @since 5.0
     */
    public abstract static class AbstractMergeBuilder<B extends AbstractMergeBuilder<B>>
                extends AbstractBuilder<B> {

        private String fromBranch;
        private String fromCommitId;
        private Repository fromRepository;
        private String toBranch;

        @Nonnull
        public B fromBranch(String value) {
            fromBranch = checkNotBlank(value, "fromBranch");

            return self();
        }

        /**
         * @param value the commit ID to merge, or {@code null} to merge the current tip of the
         *              {@link #fromBranch(String) branch}
         * @return {@code this}
         */
        @Nonnull
        public B fromCommitId(@Nullable String value) {
            fromCommitId = value;

            return self();
        }

        /**
         * Sets the {@link Repository} containing the from {@link #fromBranch(String) branch} and
         * {@link #fromCommitId(String) commit}. This value only needs to be supplied if the
         * from branch/commit are not in the same repository as the {@link #toBranch(String) target}.
         *
         * @param value the repository containing the from {@link #fromBranch(String) branch} and
         *              {@link #fromCommitId(String) commit}, if different from the target
         * @return {@code this}
         */
        @Nonnull
        public B fromRepository(@Nullable Repository value) {
            fromRepository = value;

            return self();
        }

        @Nonnull
        public B toBranch(String value) {
            toBranch = checkNotBlank(value, "toBranch");

            return self();
        }

        @Override
        protected void validate() {
            super.validate();

            if (StringUtils.isBlank(fromBranch)) {
                throw new IllegalStateException(
                        "When performing a merge, the name of the branch to merge from is required");
            }
            if (StringUtils.isBlank(toBranch)) {
                throw new IllegalStateException(
                        "When performing a merge, the name of the branch to merge into is required");
            }
        }
    }

    public static class Builder extends AbstractMergeBuilder<Builder> {

        public Builder() {
        }

        @Nonnull
        public MergeCommandParameters build() {
            validate();

            return new MergeCommandParameters(this);
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }
}
