package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.content.DiffContext;
import com.atlassian.bitbucket.content.DiffWhitespace;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Set;

import static java.util.Objects.requireNonNull;

@SuppressWarnings({"deprecation", "unused"}) //Unused members are still public API
public class AbstractDiffCommandParameters extends AbstractCommandParameters {

    public static final int DEFAULT_CONTEXT_LINES = -1;

    //See BSERV-10916 to understand how this value was chosen
    private static final int MIN_LINE_LENGTH = 100;

    private final int contextLines;
    private final int maxLineLength;
    private final int maxLines;
    private final Set<String> paths;
    private final DiffWhitespace whitespace;

    /**
     * @param maxLineLength the maximum length for any diff line
     * @param maxLines      the maximum number of diff lines to stream
     * @param contextLines  the number of context (unmodified) lines to include around changed (added/removed) lines
     * @param whitespace    controls how whitespace changes affect the diff
     * @param paths         a set containing zero or more paths to include changes for
     * @deprecated in 5.4 for removal in 6.0. Use {@link #AbstractDiffCommandParameters(AbstractBuilder)} instead.
     */
    @Deprecated
    public AbstractDiffCommandParameters(int maxLineLength, int maxLines, int contextLines, DiffWhitespace whitespace,
                                         Set<String> paths) {
        this.contextLines = contextLines;
        this.maxLineLength = maxLineLength;
        this.maxLines = maxLines;
        this.paths = paths;
        this.whitespace = whitespace;
    }

    /**
     * @param builder a builder containing the assembled parameters
     * @since 5.4
     */
    protected AbstractDiffCommandParameters(AbstractBuilder<?> builder) {
        contextLines = builder.contextLines;
        maxLineLength = Math.max(builder.maxLineLength, MIN_LINE_LENGTH); //Enforce a min read length to avoid invalid values
        maxLines = builder.maxLines;
        paths = builder.paths.build();
        whitespace = builder.whitespace;
    }

    public int getContextLines() {
        return contextLines;
    }

    public int getMaxLineLength() {
        return maxLineLength;
    }

    public int getMaxLines() {
        return maxLines;
    }

    @Nonnull
    public Set<String> getPaths() {
        return paths;
    }

    public boolean hasContextLines() {
        return contextLines != DEFAULT_CONTEXT_LINES;
    }

    public boolean hasPaths() {
        return !paths.isEmpty();
    }

    /**
     * @return optional flag to potentially ignore whitespace
     */
    @Nonnull
    public DiffWhitespace getWhitespace() {
        return whitespace;
    }

    /**
     * Appends common attributes to the provided {@link DiffContext.Builder} before building
     * it and returning the final {@link DiffContext}.
     *
     * @param builder the starting builder to apply common attributes to before building
     * @return the {@link DiffContext} from the provided builder
     */
    @Nonnull
    protected DiffContext toContext(@Nonnull DiffContext.Builder builder) {
        return requireNonNull(builder, "builder").contextLines(getContextLines())
                .maxLineLength(getMaxLineLength())
                .maxLines(getMaxLines())
                .whitespace(getWhitespace())
                .build();
    }

    @SuppressWarnings("DeprecatedIsStillUsed")
    public abstract static class AbstractBuilder<B extends AbstractBuilder<B>> {

        /**
         * @deprecated in 5.4. This field will be made {@code private} in 6.0.
         */
        @Deprecated
        protected final ImmutableSet.Builder<String> paths;

        /**
         * @deprecated in 5.4. This field will be made {@code private} in 6.0.
         */
        @Deprecated
        protected int contextLines;
        /**
         * @deprecated in 5.4. This field will be made {@code private} in 6.0.
         */
        @Deprecated
        protected int maxLineLength;
        /**
         * @deprecated in 5.4. This field will be made {@code private} in 6.0.
         */
        @Deprecated
        protected int maxLines;
        /**
         * @deprecated in 5.4. This field will be made {@code private} in 6.0.
         */
        @Deprecated
        protected DiffWhitespace whitespace;

        public AbstractBuilder() {
            contextLines = DEFAULT_CONTEXT_LINES;
            paths = ImmutableSet.builder();
            whitespace = DiffWhitespace.SHOW;
        }

        @Nonnull
        public B contextLines(int value) {
            contextLines = value;

            return self();
        }

        @Nonnull
        public B defaultContextLines() {
            contextLines = DEFAULT_CONTEXT_LINES;

            return self();
        }

        @Nonnull
        public B maxLineLength(int value) {
            maxLineLength = value;

            return self();
        }

        @Nonnull
        public B maxLines(int value) {
            maxLines = value;

            return self();
        }

        @Nonnull
        public B path(@Nullable String value) {
            addIf(NOT_BLANK, paths, value);

            return self();
        }

        @Nonnull
        public B paths(@Nullable Iterable<String> values) {
            addIf(NOT_BLANK, paths, values);

            return self();
        }

        @Nonnull
        public B paths(@Nullable String value, @Nullable String... values) {
            addIf(NOT_BLANK, paths, value, values);

            return self();
        }

        /**
         * @param value whitespace mode for showing diff
         * @return {@code this}
         */
        @Nonnull
        public B whitespace(@Nonnull DiffWhitespace value) {
            whitespace = requireNonNull(value, "whitespace");

            return self();
        }

        @Nonnull
        protected abstract B self();
    }
}
