package com.atlassian.bitbucket.repository;

import com.atlassian.bitbucket.util.BuilderSupport;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Set;

import static java.util.Objects.requireNonNull;

/**
 * Describes which refs to {@link RefService#resolveRefs(ResolveRefsRequest) resolve}.
 * <p>
 * If it is known whether a requested ref is a branch or a tag, the caller is <i>encouraged</i> to use either the
 * {@link Builder#branchId(String)} or {@link Builder#tagId(String)} method to allow the SCM to optimize resolving
 * the ref. If the ref type is unknown, the {@link Builder#refId(String)} method should be used.
 *
 * @since 4.6
 */
public class ResolveRefsRequest {

    private final Set<String> branchIds;
    private final Set<String> refIds;
    private final Repository repository;
    private final Set<String> tagIds;

    private ResolveRefsRequest(Builder builder) {
        branchIds = builder.branchIds.build();
        refIds = builder.refIds.build();
        repository = builder.repository;
        tagIds = builder.tagIds.build();
    }

    @Nonnull
    public Set<String> getBranchIds() {
        return branchIds;
    }

    @Nonnull
    public Set<String> getRefIds() {
        return refIds;
    }

    @Nonnull
    public Repository getRepository() {
        return repository;
    }

    @Nonnull
    public Set<String> getTagIds() {
        return tagIds;
    }

    public static class Builder extends BuilderSupport {

        private final ImmutableSet.Builder<String> branchIds;
        private final ImmutableSet.Builder<String> refIds;
        private final Repository repository;
        private final ImmutableSet.Builder<String> tagIds;

        public Builder(@Nonnull Repository repository) {
            this.repository = requireNonNull(repository, "repository");

            branchIds = ImmutableSet.builder();
            refIds = ImmutableSet.builder();
            tagIds = ImmutableSet.builder();
        }

        @Nonnull
        public Builder branchId(@Nullable String value) {
            addIf(NOT_BLANK, branchIds, value);

            return this;
        }

        @Nonnull
        public Builder branchIds(@Nullable String value, @Nullable String... moreValues) {
            addIf(NOT_BLANK, branchIds, value, moreValues);

            return this;
        }

        @Nonnull
        public Builder branchIds(@Nullable Iterable<String> values) {
            addIf(NOT_BLANK, branchIds, values);

            return this;
        }

        @Nonnull
        public ResolveRefsRequest build() {
            return new ResolveRefsRequest(this);
        }

        @Nonnull
        public Builder refId(@Nullable String value) {
            addIf(NOT_BLANK, refIds, value);

            return this;
        }

        @Nonnull
        public Builder refIds(@Nullable String value, @Nullable String... moreValues) {
            addIf(NOT_BLANK, refIds, value, moreValues);

            return this;
        }

        @Nonnull
        public Builder refIds(@Nullable Iterable<String> values) {
            addIf(NOT_BLANK, refIds, values);

            return this;
        }

        @Nonnull
        public Builder tagId(@Nullable String value) {
            addIf(NOT_BLANK, tagIds, value);

            return this;
        }

        @Nonnull
        public Builder tagIds(@Nullable String value, @Nullable String... moreValues) {
            addIf(NOT_BLANK, tagIds, value, moreValues);

            return this;
        }

        @Nonnull
        public Builder tagIds(@Nullable Iterable<String> values) {
            addIf(NOT_BLANK, tagIds, values);

            return this;
        }
    }
}
