package com.atlassian.bitbucket.pull;

import javax.annotation.Nonnull;
import java.util.Set;

/**
 * Describes the configuration for how pull requests will be merged in a given repository. For
 * {@link PullRequest#isCrossRepository cross-repository} pull requests, the {@link PullRequest#getToRef target
 * repository}'s configuration controls merge behavior.
 * <p>
 * The SCM backing each repository may expose one or more strategies that are available when merging pull requests.
 * For SCMs which support multiple strategies, administrators can configure which strategy (or strategies) should
 * be available. Any {@link PullRequestMergeStrategy#isEnabled() enabled} strategy may be selected when
 * {@link PullRequestService#merge merging} a pull request. If an explicit strategy is not requested, the
 * {@link #getDefaultStrategy() default} will be used.
 * <p>
 * The strategies to use may be configured for a specific repository, all repositories in a given project which
 * share the same SCM, or for all repositories of a specific SCM in all projects. If configuration has not been
 * applied at <i>any</i> of those levels, defaults provided by the SCM are used. The {@link #getType type} can be
 * used to determine what level configuration was applied at.
 *
 * @since 4.9
 */
public interface PullRequestMergeConfig {

    /**
     * Retrieves the default strategy, which will be used automatically when merging pull requests if an
     * {@link PullRequestMergeRequest#getStrategyId() explicit strategy} is not requested. The default
     * strategy is always {@link PullRequestMergeStrategy#isEnabled enabled}.
     *
     * @return the default merge strategy
     */
    @Nonnull
    PullRequestMergeStrategy getDefaultStrategy();

    /**
     * Retrieves the <i>complete set</i> of merge strategies supported by the SCM, including the
     * {@link #getDefaultStrategy() default strategy}. Some of the returned strategies may not be
     * {@link PullRequestMergeStrategy#isEnabled() enabled}. Only enabled strategies may be
     * {@link PullRequestMergeRequest#getStrategyId() requested} when merging pull requests, so
     * callers should check before using a strategy from the returned set.
     *
     * @return the complete set of <i>supported</i> strategies for the backing SCM, including the
     *         {@link #getDefaultStrategy() default}, each of which may or may not be
     *         {@link PullRequestMergeStrategy#isEnabled enabled}
     */
    @Nonnull
    Set<PullRequestMergeStrategy> getStrategies();

    /**
     * Retrieves the configuration type, identifying the level the configuration was applied at. Since configuration
     * is inherited, the type may be <i>less specific</i> than what was requested.
     *
     * @return the configuration type, used to determine what level configuration was applied at
     */
    @Nonnull
    PullRequestMergeConfigType getType();
}
