package com.atlassian.bitbucket.pull;

import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.util.BuilderSupport;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;
import static org.apache.commons.lang3.StringUtils.trimToNull;

/**
 * A request to create a new pull request using the provided properties.
 * 
 * @since 5.13
 */
public class PullRequestCreateRequest {

    private final String description;
    private final String fromRefId;
    private final Repository fromRepository;
    private final Set<String> reviewers;
    private final String title;
    private final String toBranchId;
    private final Repository toRepository;

    protected PullRequestCreateRequest(@Nonnull AbstractBuilder<?> builder) {
        title = requireNonNull(builder.title, "title");
        description = builder.description;
        fromRepository = requireNonNull(builder.fromRepository, "fromRepository");
        fromRefId = requireNonNull(builder.fromBranchId, "fromRefId");
        reviewers = builder.reviewers.build();
        toRepository = requireNonNull(builder.toRepository, "toRepository");
        toBranchId = requireNonNull(builder.toBranchId, "toBranchId");
    }

    @Nonnull
    public Optional<String> getDescription() {
        return ofNullable(description);
    }

    @Nonnull
    public String getFromRefId() {
        return fromRefId;
    }

    @Nonnull
    public Repository getFromRepository() {
        return fromRepository;
    }

    @Nonnull
    public Set<String> getReviewers() {
        return reviewers;
    }

    @Nonnull
    public String getTitle() {
        return title;
    }

    @Nonnull
    public String getToBranchId() {
        return toBranchId;
    }

    @Nonnull
    public Repository getToRepository() {
        return toRepository;
    }

    public static class Builder extends AbstractBuilder<Builder> {

        public Builder() {
        }

        @Nonnull
        @Override
        public PullRequestCreateRequest build() {
            return new PullRequestCreateRequest(this);
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }

    public abstract static class AbstractBuilder<T extends AbstractBuilder<T>> extends BuilderSupport {

        private String description;
        private String fromBranchId;
        private Repository fromRepository;
        private ImmutableSet.Builder<String> reviewers;
        private String title;
        private String toBranchId;
        private Repository toRepository;

        protected AbstractBuilder() {
            reviewers = ImmutableSet.builder();
        }

        @Nonnull
        public abstract PullRequestCreateRequest build();

        @Nonnull
        public T description(@Nullable String value) {
            description = trimToNull(value);
            return self();
        }

        @Nonnull
        public T fromRefId(@Nonnull String value) {
            fromBranchId = requireNonNull(trimToNull(value), "fromRefId");
            return self();
        }

        @Nonnull
        public T fromRepository(@Nonnull Repository value) {
            fromRepository = requireNonNull(value, "fromRepository");
            return self();
        }

        @Nonnull
        public T repository(@Nonnull Repository value) {
            fromRepository = toRepository = requireNonNull(value, "repository");
            return self();
        }

        @Nonnull
        public T reviewer(@Nonnull String value) {
            addIf(Objects::nonNull, reviewers, value);
            return self();
        }

        @Nonnull
        public T reviewers(@Nonnull Set<String> value) {
            addIf(Objects::nonNull, reviewers, value);
            return self();
        }

        @Nonnull
        public T title(@Nonnull String value) {
            title = requireNonNull(trimToNull(value), "title");
            return self();
        }

        @Nonnull
        public T toBranchId(@Nonnull String value) {
            toBranchId = requireNonNull(trimToNull(value), "toBranchId");
            return self();
        }

        @Nonnull
        public T toRepository(@Nonnull Repository value) {
            toRepository = requireNonNull(value, "toRepository");
            return self();
        }

        /**
         * Overridden in concrete builder implementations to return {@code this}.
         *
         * @return {@code this}
         */
        @Nonnull
        protected abstract T self();
    }
}
