package com.atlassian.bitbucket.pull;

import com.atlassian.bitbucket.util.BuilderSupport;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Objects;
import java.util.Set;

/**
 * Request for searching for pull requests associated with a commit.
 *
 * @since 5.11
 */
public class PullRequestCommitSearchRequest {

    private final String commitId;
    private final Set<PullRequestState> states;
    private final Integer toRepositoryId;

    private PullRequestCommitSearchRequest(Builder builder) {
        commitId = builder.commitId;
        states = builder.states.build();
        toRepositoryId = builder.toRepositoryId;
    }

    /**
     * Limits the search to pull requests that contain or created (in the case of a rebase/squash merge) the
     * specified commit ID.
     *
     * @return the commit ID
     */
    @Nonnull
    public String getCommitId() {
        return commitId;
    }

    /**
     * When set, limits the search to pull requests in <i>any</i> of the specified {@link PullRequestState states}.
     *
     * @return a set containing <i>zero</i> or more states to filter by
     */
    @Nonnull
    public Set<PullRequestState> getStates() {
        return states;
    }

    /**
     * When set, limits the search to pull requests which {@link PullRequest#getToRef target} the specified repository.
     *
     * @return the target repository ID
     */
    @Nullable
    public Integer getToRepositoryId() {
        return toRepositoryId;
    }

    public static class Builder extends BuilderSupport {

        private final String commitId;
        private final ImmutableSet.Builder<PullRequestState> states;

        private Integer toRepositoryId;

        /**
         * Creates a new builder that limits pull requests to those that contain or created (via their merge) the
         * specified commit.
         */
        public Builder(@Nonnull String commitId) {
            this.commitId = requireNonBlank(commitId, "commitId");

            states = ImmutableSet.builder();
        }

        /**
         * Assembles a new {@link PullRequestCommitSearchRequest} from the provided values.
         *
         * @return a new search request based on the provided criteria
         */
        @Nonnull
        public PullRequestCommitSearchRequest build() {
            return new PullRequestCommitSearchRequest(this);
        }

        /**
         * Limits the search to pull requests in the specified {@link PullRequest#getState state}.
         *
         * @param value the state, which is <i>ignored</i> if it is {@code null}
         * @return {@code this}
         */
        @Nonnull
        public Builder state(@Nullable PullRequestState value) {
            addIf(Objects::nonNull, states, value);

            return this;
        }

        /**
         * Limits the search to pull requests in <i>any</i> of the specified {@link PullRequest#getState states}.
         *
         * @param values the allowed states, from which any {@code null} entries are ignored
         * @return {@code this}
         */
        @Nonnull
        public Builder states(@Nullable Iterable<PullRequestState> values) {
            addIf(Objects::nonNull, states, values);

            return this;
        }

        /**
         * Limits the search to pull requests in <i>any</i> of the specified {@link PullRequest#getState states}.
         *
         * @param value  the first state, which is <i>ignored</i> if it is {@code null}
         * @param values additional states, from which any {@code null} entries are ignored
         * @return {@code this}
         */
        @Nonnull
        public Builder states(@Nullable PullRequestState value, @Nullable PullRequestState... values) {
            addIf(Objects::nonNull, states, value, values);

            return this;
        }

        /**
         * Limits the search to pull requests {@link PullRequest#getToRef targeting} the specified repository.
         *
         * @param toRepositoryId the target repository ID
         * @return {@code this}
         */
        @Nonnull
        public Builder toRepositoryId(@Nullable Integer toRepositoryId) {
            this.toRepositoryId = toRepositoryId;
            return this;
        }
    }
}
