package com.atlassian.bitbucket.property;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.*;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Immutable collection of properties mapped by string keys. Allows for storing any object as a property value.
 * <p>
 * It is strongly encouraged that the property values can be marshalled to/from JSON, or they will not be included in
 * REST responses.
 *
 * @see PropertySupport
 */
public class PropertyMap implements Map<String, Object> {

    public static final PropertyMap EMPTY = new PropertyMap(Collections.<String, Object>emptyMap());

    private final Map<String, Object> delegate;

    private PropertyMap(Map<String, Object> delegate) {
        this.delegate = delegate;
    }

    @Nullable
    public <T> T getAs(@Nonnull String key, @Nonnull Class<T> expectedType) {
        checkNotNull(key, "key");
        checkNotNull(expectedType, "expectedType");

        return expectedType.cast(get(key));
    }

    /**
     * Convenience method to return the property value as an instance of {@code Iterable} with element type {@code E}.
     *
     * @param key property key
     * @param iterableType class representing the expected type of the iterable
     * @param elementType class representing the expected type of the elements
     * @return property value, as an instance of {@code I}, or {@code null}, if this property map does not contain a
     * property with key {@code key}
     * @throws ClassCastException if the property exists, but is not of expected type {@code iterableType}
     */
    @Nullable
    public <E, I extends Iterable<E>> I getAs(@Nonnull String key, @Nonnull Class<I> iterableType,
                                              @Nonnull Class<E> elementType) {
        checkNotNull(key, "key");
        checkNotNull(iterableType, "iterableType");
        checkNotNull(elementType, "elementType");

        return iterableType.cast(get(key));
    }

    // Map API

    @Override
    public int size() {
        return delegate.size();
    }

    @Override
    public boolean isEmpty() {
        return delegate.isEmpty();
    }

    @Override
    public boolean containsKey(Object key) {
        return delegate.containsKey(key);
    }

    @Override
    public boolean containsValue(Object value) {
        return delegate.containsValue(value);
    }

    @Override
    public Object get(Object key) {
        return delegate.get(key);
    }

    @Override
    public Object put(String key, Object value) {
        throw new UnsupportedOperationException("PropertiesMap is immutable");
    }

    @Override
    public Object remove(Object key) {
        throw new UnsupportedOperationException("PropertiesMap is immutable");
    }

    @Override
    public void putAll(@Nonnull Map<? extends String, ?> other) {
        throw new UnsupportedOperationException("PropertiesMap is immutable");
    }

    @Override
    public void clear() {
        throw new UnsupportedOperationException("PropertiesMap is immutable");
    }

    @Override
    @Nonnull
    public Set<String> keySet() {
        return delegate.keySet();
    }

    @Override
    @Nonnull
    public Collection<Object> values() {
        return delegate.values();
    }

    @Override
    @Nonnull
    public Set<Map.Entry<String, Object>> entrySet() {
        return delegate.entrySet();
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }
        if (obj instanceof PropertyMap) {
            return delegate.equals(((PropertyMap) obj).delegate);
        }

        return false;
    }

    @Override
    public int hashCode() {
        return delegate.hashCode();
    }

    @Override
    public String toString() {
        return delegate.toString();
    }

    public static class Builder {

        private final HashMap<String, Object> map = new HashMap<>();

        public Builder property(@Nonnull String key, @Nonnull Object value) {
            map.put(key, value);

            return this;
        }

        public Builder properties(@Nonnull Map<String, ?> properties) {
            map.putAll(checkNotNull(properties, "properties"));

            return this;
        }

        public PropertyMap build() {
            return new PropertyMap(Collections.unmodifiableMap(map));
        }
    }
}
