package com.atlassian.bitbucket.permission;

import com.atlassian.bitbucket.project.Project;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.user.ApplicationUser;
import com.atlassian.bitbucket.user.UserType;
import com.atlassian.bitbucket.util.Page;
import com.atlassian.bitbucket.util.PageRequest;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

/**
 * Updates and queries the permissions of users and groups.
 *
 * @see PermissionService
 */
public interface PermissionAdminService {

    /**
     * Check if a {@link UserType#NORMAL normal} user can remove themselves from the given group.
     * @param username the user to check
     * @param group the group from which the user is to be removed.
     */
    void canRemoveUserFromGroup(@Nonnull String username, @Nonnull String group);

    /**
     * Check if the current user can add users to the given group.
     * @param group the group to which users will be added.
     */
    void canAddUserToGroup(@Nonnull String group);

    /**
     * Check if the current user can delete a group without affecting their permissions
     *
     * @param group the group to be deleted.
     */
    void canDeleteGroup(@Nonnull String group);

    /**
     * Check if the {@link UserType#NORMAL normal} user can be deleted.
     *
     * @param userName the name of the user being deleted
     */
    void canDeleteUser(@Nonnull String userName);

    /**
     * Retrieves a page of {@link PermittedGroup groups} and their highest global permission. Groups which do not have any global permissions
     * <i>explicitly</i> assigned to them will not be returned.
     *
     * @param filter if non-null non-empty then the groups returned must include this text
     * @param pageRequest bounds the page of groups to be returned
     * @return a page containing zero or more groups which have been explicitly granted a global permission, and their
     *         highest granted permission
     */
    @Nonnull
    Page<PermittedGroup> findGroupsWithGlobalPermission(@Nullable String filter, @Nonnull PageRequest pageRequest);

    /**
     * Retrieves a page of groups who have not been explicitly granted any global permission
     *
     * @param filter if non-null non-empty then the groups returned must include this text
     * @param pageRequest bounds the page of groups to be returned
     * @return a page of groupnames.
     */
    @Nonnull
    Page<String> findGroupsWithoutGlobalPermission(@Nullable String filter, @Nonnull PageRequest pageRequest);

    /**
     * Retrieves a page of groups who have not been explicitly granted any project permission
     *
     * @param project         the project for which the permissions are required.
     * @param filter          if non-null non-empty then the groups returned must include this text
     * @param pageRequest     bounds the page of groups to be returned
     * @return a page of groupnames.
     */
    @Nonnull
    Page<String> findGroupsWithoutProjectPermission(@Nonnull Project project,
                                                    @Nullable String filter,
                                                    @Nonnull PageRequest pageRequest);


    /**
     * Retrieves a page of groups who have not been explicitly granted any repository permission.
     *
     * @param repository      the repository for which the permissions are required.
     * @param filter          if non-null non-empty then the groups returned must include this text
     * @param pageRequest     bounds the page of groups to be returned
     * @return a page of groupnames.
     */
    @Nonnull
    Page<String> findGroupsWithoutRepositoryPermission(@Nonnull Repository repository,
                                                       @Nullable String filter,
                                                       @Nonnull PageRequest pageRequest);

    /**
     * Retrieves a page of {@link PermittedGroup groups} and their highest granted permission for the specified project.
     * Groups which do not have any permissions <i>explicitly</i> assigned to them for the specified
     * project will not be returned.
     *
     * @param project     the project for which permissions should be considered
     * @param filter      if non-null non-empty then the groups returned must include this text
     * @param pageRequest bounds the page of groups to be returned
     * @return a page containing zero or more groups which have been explicitly granted a permission for the specified
     *         project, and their highest granted permission
     */
    @Nonnull
    Page<PermittedGroup> findGroupsWithProjectPermission(@Nonnull Project project,
                                                         @Nullable String filter,
                                                         @Nonnull PageRequest pageRequest);

    /**
     * Retrieves a page of {@link PermittedGroup groups} and their highest granted permission for the specified
     * repository. Groups which do not have any permissions <i>explicitly</i> assigned to them for the specified
     * repository will not be returned.
     *
     * @param repository  the repository for which permissions should be considered
     * @param filter      if non-null non-empty then the groups returned must include this text
     * @param pageRequest bounds the page of groups to be returned
     * @return a page containing zero or more groups which have been explicitly granted a permission for the specified
     *         repository, and their highest granted permission
     */
    @Nonnull
    Page<PermittedGroup> findGroupsWithRepositoryPermission(@Nonnull Repository repository,
                                                            @Nullable String filter,
                                                            @Nonnull PageRequest pageRequest);

    /**
     * Retrieves a page of {@link ApplicationUser#isActive() active} and {@link Permission#LICENSED_USER licensed} users who
     * have not been explicitly granted any project permission
     *
     * @param project         the project for which the permissions are required.
     * @param filter          if non-null non-empty then the usernames returned must include this text
     * @param pageRequest     bounds the page of users to be returned
     * @return a page of ApplicationUser instances.
     */
    @Nonnull
    Page<ApplicationUser> findLicensedUsersWithoutProjectPermission(@Nonnull Project project,
                                                              @Nullable String filter,
                                                              @Nonnull PageRequest pageRequest);

    /**
     * Retrieves a page of {@link ApplicationUser#isActive() active} and {@link Permission#LICENSED_USER licensed} users who
     * have not been explicitly granted any repository permission.
     *
     * @param repository      the repository for which the permissions are required.
     * @param filter          if non-null non-empty then the usernames returned must include this text
     * @param pageRequest     bounds the page of users to be returned
     * @return a page of ApplicationUser instances.
     */
    @Nonnull
    Page<ApplicationUser> findLicensedUsersWithoutRepositoryPermission(@Nonnull Repository repository,
                                                                 @Nullable String filter,
                                                                 @Nonnull PageRequest pageRequest);

    /**
     * Retrieves a page of {@link PermittedUser users} and their highest global permission. Users which do not have any
     * global permissions <i>explicitly</i> assigned to them will not be returned.
     *
     * @param filter if non-null non-empty then the usernames returned must include this text
     * @param pageRequest bounds the page of users to be returned
     * @return a page containing zero or more users who have been explicitly granted a global permission, and their
     *         highest granted permission
     */
    @Nonnull
    Page<PermittedUser> findUsersWithGlobalPermission(@Nullable String filter, @Nonnull PageRequest pageRequest);

    /**
     * Retrieves a page of {@link ApplicationUser#isActive() active} users who have not been explicitly
     * granted any global permission
     *
     * @param filter if non-null non-empty then the usernames returned must include this text
     * @param pageRequest bounds the page of users to be returned
     * @return a page of ApplicationUser instances.
     */
    @Nonnull
    Page<ApplicationUser> findUsersWithoutGlobalPermission(@Nullable String filter, @Nonnull PageRequest pageRequest);

    /**
     * Retrieves a page of {@link PermittedUser users} and their highest granted permission for the specified project. Users
     * which do not have any permissions <i>explicitly</i> assigned to them for the specified project will not be
     * returned.
     *
     * @param project     the project for which permissions should be considered
     * @param filter      if non-null non-empty then the usernames returned must include this text
     * @param pageRequest bounds the page of users to be returned
     * @return a page containing zero or more users who have been explicitly granted a permission for the specified
     *         project, and their highest granted permission
     */
    @Nonnull
    Page<PermittedUser> findUsersWithProjectPermission(@Nonnull Project project,
                                                       @Nullable String filter,
                                                       @Nonnull PageRequest pageRequest);

    /**
     * Retrieves a page of {@link PermittedUser users} and their highest granted permission for the specified
     * repository. Users which do not have any permissions <i>explicitly</i> assigned to them for the specified
     * repository will not be returned.
     *
     * @param repository  the repository for which permissions should be considered
     * @param filter      if non-null non-empty then the usernames returned must include this text
     * @param pageRequest bounds the page of users to be returned
     * @return a page containing zero or more users who have been explicitly granted a permission for the specified
     *         repository, and their highest granted permission
     */
    @Nonnull
    Page<PermittedUser> findUsersWithRepositoryPermission(@Nonnull Repository repository,
                                                          @Nullable String filter,
                                                          @Nonnull PageRequest pageRequest);

    /**
     * Assigns a permission to multiple users and/or groups.
     * <p>
     * Note that:
     * <ul>
     *     <li>granting a global permission to users or groups will remove their other <em>global</em> permissions,</li>
     *     <li>granting a project permission to users or groups will remove their other <em>project</em> permissions on the same project,</li>
     *     <li>granting a repository permission to users or groups will remove their other <em>repository</em> permissions on the same repository.</li>
     * </ul>
     *
     * @param request request specifying which permission to grant to which users and/or groups
     * @see #grantAllProjectPermission(Permission, Project) to grant default project permissions
     */
    void setPermission(@Nonnull SetPermissionRequest request);

    /**
     * Revoke all global permissions for the given group.
     * @param group name of the group
     */
    void revokeAllGlobalPermissions(@Nonnull String group);

    /**
     * Revoke all project permissions for the given group.
     * @param project the project on which permissions are to be revoked
     * @param group name of the group
     */
    void revokeAllProjectPermissions(@Nonnull Project project, @Nonnull String group);

    /**
     * Revoke all repository permissions for the given group.
     * @param repository the repository on which permissions are to be revoked
     * @param group name of the group
     */
    void revokeAllRepositoryPermissions(@Nonnull Repository repository, @Nonnull String group);

    /**
     * Revoke all global permissions for the given user.
     * @param user user to revoke permissions from
     */
    void revokeAllGlobalPermissions(@Nonnull ApplicationUser user);

    /**
     * Revoke all project permissions for the given user.
     * @param project the project on which permissions are to be revoked
     * @param user user to revoke permissions from
     */
    void revokeAllProjectPermissions(@Nonnull Project project, @Nonnull ApplicationUser user);

    /**
     * Revoke all repository permissions for the given user.
     * @param repository the repository on which permissions are to be revoked
     * @param user user to revoke permissions from
     */
    void revokeAllRepositoryPermissions(@Nonnull Repository repository, @Nonnull ApplicationUser user);

    /**
     * Check if the given project permission has been granted to every logged in user for the given project.
     *
     * @param permission the permission to be checked.
     * @param project the project for which the permissions are required.
     * @return true if the given global permission is granted to all users.
     */
    boolean hasAllProjectPermission(@Nonnull Permission permission, @Nonnull Project project);

    /**
     * Grant a project permission to all users users
     *
     * @param permission the project permission to be granted
     * @param project the project on which permission is to be revoked
     */
    void grantAllProjectPermission(@Nonnull Permission permission, @Nonnull Project project);

    /**
     * Revoked a project permission from all users users
     *
     * @param permission the project permission to be revoked
     * @param project the project on which permission is to be revoked
     */
    void revokeAllProjectPermission(@Nonnull Permission permission, @Nonnull Project project);

    /**
     * Revoke all permissions from a {@link UserType#NORMAL normal} user.
     * {@link UserType#SERVICE service} users can not be identified by username hence their
     * permission cannot be revoked by this method.
     *
     * @param name name of the normal user
     */
    void revokeAllUserPermissions(@Nonnull String name);

    /**
     * Revoke all permissions from a user.
     *
     * @param user the user
     */
    void revokeAllUserPermissions(@Nonnull ApplicationUser user);

    /**
     * Revoke all permissions granted to a group.
     *
     * @param name name of the group
     */
    void revokeAllGroupPermissions(@Nonnull String name);
}
