package com.atlassian.bitbucket.event.repository;

import com.atlassian.bitbucket.event.CancelableEvent;
import com.atlassian.bitbucket.i18n.KeyedMessage;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.repository.RepositoryService;
import com.atlassian.bitbucket.repository.RepositoryUpdateRequest;
import com.atlassian.bitbucket.util.CancelState;

import javax.annotation.Nonnull;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Raised just before a {@link Repository repository} is updated. This event is synchronous, allowing listeners to
 * perform operations in the same database transaction where the repository will be updated.
 * <p>
 * When this event is raised, the repository's persisted state has <i>not</i> been updated, so {@link #getRepository()}
 * matches the {@link #getOldValue() old values}.
 * <p>
 * This event is {@link CancelableEvent cancelable}. A listener may prevent the repository from being updated by
 * {@link #cancel(KeyedMessage) canceling} this event. Throwing an exception <i>will not</i> prevent the repository
 * from being updated; the exception will be logged and ignored.
 *
 * @see RepositoryService#update(RepositoryUpdateRequest)
 */
public class RepositoryModificationRequestedEvent
        extends AbstractRepositoryModificationEvent
        implements CancelableEvent {

    private final CancelState cancelState;

    public RepositoryModificationRequestedEvent(@Nonnull Object source, @Nonnull Repository oldValue,
                                                @Nonnull Repository newValue, @Nonnull CancelState cancelState) {
        super(source, oldValue, oldValue, newValue);

        this.cancelState = checkNotNull(cancelState, "cancelState");
    }

    /**
     * Cancels repository modification, providing a message explaining why.
     *
     * @param message a descriptive message explaining why the operation has been canceled
     */
    @Override
    public void cancel(@Nonnull KeyedMessage message) {
        cancelState.cancel(message);
    }

    /**
     * Retrieves a flag indicating whether repository modification has already been canceled by another listener.
     *
     * @return {@code true} if another listener has already canceled repository modification; otherwise, {@code false}
     */
    @Override
    public boolean isCanceled() {
        return cancelState.isCanceled();
    }
}
