package com.atlassian.bitbucket.event.pull;

import com.atlassian.bitbucket.event.CancelableEvent;
import com.atlassian.bitbucket.event.annotation.TransactionAware;
import com.atlassian.bitbucket.i18n.KeyedMessage;
import com.atlassian.bitbucket.pull.PullRequest;
import com.atlassian.bitbucket.pull.PullRequestAction;
import com.atlassian.bitbucket.pull.PullRequestDeleteRequest;
import com.atlassian.bitbucket.pull.PullRequestService;
import com.atlassian.bitbucket.util.CancelState;

import javax.annotation.Nonnull;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Raised just before a {@link PullRequest pull request} is {@link PullRequestDeletedEvent deleted}. This event is
 * synchronous, allowing listeners to perform operations in the same database transaction where the pull request
 * will be deleted.
 * <p>
 * This event is {@link CancelableEvent cancelable}. A listener may prevent the pull request from being deleted by
 * {@link #cancel(KeyedMessage) canceling} this event. Throwing an exception <i>will not</i> prevent the pull request
 * from being deleted; the exception will be logged and ignored.
 *
 * @see PullRequestService#delete(PullRequestDeleteRequest)
 * @since 5.1
 */
@TransactionAware(TransactionAware.When.IMMEDIATE) //Override transactionality from PullRequestEvent
public class PullRequestDeletionRequestedEvent extends PullRequestEvent implements CancelableEvent {

    private final CancelState cancelState;

    public PullRequestDeletionRequestedEvent(@Nonnull Object source, @Nonnull PullRequest pullRequest,
                                             @Nonnull CancelState cancelState) {
        super(source, pullRequest, PullRequestAction.DELETED);

        this.cancelState = checkNotNull(cancelState, "cancelState");
    }

    /**
     * Cancels pull request deletion, providing a message explaining why.
     *
     * @param message a descriptive message explaining why the operation has been canceled
     */
    @Override
    public void cancel(@Nonnull KeyedMessage message) {
        cancelState.cancel(message);
    }

    /**
     * Retrieves a flag indicating whether pull request creation has already been canceled by another listener.
     *
     * @return {@code true} if another listener has already canceled pull request creation; otherwise, {@code false}
     */
    @Override
    public boolean isCanceled() {
        return cancelState.isCanceled();
    }
}
