package com.atlassian.bitbucket.event.pull;

import com.atlassian.bitbucket.comment.Comment;
import com.atlassian.bitbucket.comment.CommentAction;
import com.atlassian.bitbucket.pull.PullRequest;
import com.atlassian.bitbucket.pull.PullRequestAction;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Event that is raised when a comment is added, edited or deleted on a pull request.
 */
public abstract class PullRequestCommentEvent extends PullRequestEvent {

    private final Comment comment;
    private final Comment parent;
    private final CommentAction commentAction;

    /**
     * Constructs a new {@code PullRequestCommentEvent}.
     *
     * @param source        the component raising the event
     * @param pullRequest   the affected pull request
     * @param comment       the affected comment
     * @param parent        the parent of the comment, which will be {@code null} for top-level comments
     * @param commentAction the action performed on the comment
     */
    protected PullRequestCommentEvent(@Nonnull Object source, @Nonnull PullRequest pullRequest, @Nonnull Comment comment,
                                      @Nullable Comment parent, @Nonnull CommentAction commentAction) {
        super(source, pullRequest, PullRequestAction.COMMENTED);

        this.comment = checkNotNull(comment, "comment");
        this.parent = parent;
        this.commentAction = checkNotNull(commentAction, "commentAction");
    }

    @Nonnull
    public Comment getComment() {
        return comment;
    }

    @Nullable
    public Comment getParent() {
        return parent;
    }

    @Nonnull
    public CommentAction getCommentAction() {
        return commentAction;
    }
}
