package com.atlassian.bitbucket.event.cluster;

import com.atlassian.bitbucket.cluster.ClusterNode;
import com.atlassian.bitbucket.event.ApplicationEvent;

import javax.annotation.Nonnull;
import java.util.Date;
import java.util.EventObject;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * A base class for events related to the cluster.
 * <p>
 * This class exists primarily to simplify writing event types. Plugins should not listen for this low-level base
 * class; they should listen for specific subclasses.
 * <p>
 * Cluster node events are not part of the {@link ApplicationEvent ApplicationEvent} hierarchy. Most cluster
 * node events happen in response to system-level actions, like new nodes joining or existing nodes departing, rather than
 * happening in response to user actions, so they have their own hierarchy.
 */
public abstract class ClusterNodeEvent extends EventObject {

    private final long date;
    private final ClusterNode node;

    ClusterNodeEvent(@Nonnull Object source, @Nonnull ClusterNode node) {
        super(checkNotNull(source, "source"));

        this.node = checkNotNull(node, "node");

        date = System.currentTimeMillis();
    }

    /**
     * @return the timestamp when the event was <i>raised</i>
     */
    @Nonnull
    public Date getDate() {
        return new Date(date); //Return a new date to ensure immutability for the event instance
    }

    /**
     * Retrieves the node for the event. <i>This is intentionally not public.</i> Derived types should expose the
     * node using a more specific name that makes sense for the context rather than this generic name.
     *
     * @return the event node
     */
    @Nonnull
    ClusterNode getNode() {
        return node;
    }
}
