package com.atlassian.bitbucket.content;

import com.atlassian.bitbucket.ServiceException;
import com.atlassian.bitbucket.commit.Commit;
import com.atlassian.bitbucket.hook.repository.RepositoryHookVetoedException;
import com.atlassian.bitbucket.io.TypeAwareOutputSupplier;
import com.atlassian.bitbucket.repository.NoSuchBranchException;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.scm.FeatureUnsupportedScmException;
import com.atlassian.bitbucket.util.Page;
import com.atlassian.bitbucket.util.PageRequest;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

/**
 * A service for the management of files and their metadata.
 */
public interface ContentService {

    /**
     * Adds a new file or updates the content of an existing file.
     *
     * @param request details for adding/updating a file
     * @return the commit containing the file edit
     * @throws EditFileFailedException        if the edit operation fails
     * @throws FeatureUnsupportedScmException if the SCM for the specified repository does not support editing files
     * @throws FileAlreadyExistsException     if a file or path exists at the given path on the tip of the branch and
     *                                        no source commit was given
     * @throws FileContentUnmodifiedException if the given content is identical to the content of the file at the tip
     *                                        of the given branch
     * @throws RepositoryHookVetoedException  if one or more hooks canceled the {@link FileEditHookRequest file edit}
     * @throws FileOutOfDateException         if the given file has been changed since the source commit occurred
     * @throws FileTranscodingException       if the input content could not be transcoded to the existing
     *                                        file's encoding
     * @throws NoSuchBranchException          if the given branch does not exist in the repository
     * @throws NoSuchPathException            if the file is not present in the source commit <i>or</i> at the tip of
     *                                        the given branch
     * @since 4.13
     */
    Commit editFile(@Nonnull EditFileRequest request) throws ServiceException;

    /**
     * Returns {@link Blame} for the lines included by the provided {@link PageRequest} for the specified objectId+path
     * in the specified repository.
     * <p>
     * The objectId provided here can be several things:
     * <ul>
     *     <li>A branch name</li>
     *     <li>A tag name</li>
     *     <li>A hash (which can identify a branch, a tag, or a specific commit, among other things)</li>
     * </ul>
     *
     * @param repository  the repository to use
     * @param objectId    an identifier (name or hash) for the branch, tag or commit at which blame should be retrieved
     * @param path        the path for which blame should be retrieved
     * @param pageRequest defines the set of lines within the file for which blame should be generated
     * @return a list containing 0 or more {@link Blame} objects, ordered by the line at which they start in the file
     *
     * @since 5.0
     */
    @Nonnull
    Page<Blame> getBlame(@Nonnull Repository repository, @Nonnull String objectId, @Nonnull String path,
                         @Nonnull PageRequest pageRequest);

    /**
     * Retrieves the maximum size, in bytes, for uploading content. Files larger than this will be rejected by
     * the server.
     *
     * @return the maximum size, in bytes, for uploading content
     * @since 4.13
     */
    long getMaxUploadSize();

    @Nonnull
    ContentTreeNode.Type getType(@Nonnull Repository repository, @Nonnull String objectId, @Nullable String path);

    /**
     * Streams an archive of the specified repository at the specified commit, optionally filtering by path. The
     * content type for the requested archive format will be provided to the supplier when retrieving the output
     * stream to which the archive will be written.
     *
     * @param request        describes the commit to archive, the format to archive it in, and the repository to
     *                       archive it from
     * @param outputSupplier a supplier which, when given the archive content type, will provide an output stream
     * @since 5.1
     */
    void streamArchive(@Nonnull ArchiveRequest request, @Nonnull TypeAwareOutputSupplier outputSupplier);

    /**
     * Performs a directory listing for the specified objectId+path from the specified repository, streaming the results
     * to the provided {@code callback}.
     * <p>
     * The objectId provided here can be several things:
     * <ul>
     *     <li>A branch name</li>
     *     <li>A tag name</li>
     *     <li>A hash (which can identify a branch, a tag, or a specific commit, among other things)</li>
     * </ul>
     * <p>
     * The listing is provided to the caller via a callback mechanism - the supplied {@link ContentTreeCallback}
     * parameter. This method promises to call {@link ContentTreeCallback#onStart(ContentTreeContext)} on the supplied instance
     * once followed by zero or more {@link ContentTreeCallback#onTreeNode(ContentTreeNode)}
     * for each path in the directory  and finally followed by a {@link ContentTreeCallback#onEnd(ContentTreeSummary)}.
     * <p>
     * When {@code recursive} is {@code false}, {@link File#getSize() file sizes} will be populated and subdirectories
     * below the specified {@code path} will be returned. For recursive listings, subdirectories and file sizes are
     * omitted to improve performance.
     *
     * @param repository  the repository to use
     * @param objectId    an identifier (name or hash) for the branch, tag or commit at which blame should be retrieved
     * @param path        the path for which blame should be retrieved
     * @param recursive   whether the directory listing should be recursive
     * @param callback    the callback instance to call as the page starts, the children nodes are encountered and as the page ends
     * @param pageRequest defines the range of children in the listing the caller is interested in
     */
    void streamDirectory(@Nonnull Repository repository, @Nonnull String objectId, @Nullable String path,
                         boolean recursive, @Nonnull ContentTreeCallback callback, @Nonnull PageRequest pageRequest);

    /**
     * @param repository the repository
     * @param objectId the identifier for a point in the repository. Can be a commit ID, a branch or a tag
     * @param path the path to the file
     * @param pageRequest the page request for the start and limit of the page
     * @param withBlame {@code true} to include blame information for content; otherwise, {@code false} to omit it
     * @param fileContentCallback the callback for receiving lines and page information
     * @throws ServiceException if the stream file operation fails
     */
    void streamFile(@Nonnull Repository repository, @Nonnull String objectId, @Nonnull String path,
                    @Nonnull PageRequest pageRequest, boolean withBlame,
                    @Nonnull FileContentCallback fileContentCallback) throws ServiceException;

    /**
     * Streams the raw content of the file into an OutputStream obtained from the given TypeAwareOutputSupplier.
     * The implementation might try to detect the mime-type of the file, and pass that information to the
     * TypeAwareOutputSupplier.
     *
     * Note: this method is not paged.
     *
     * @param repository the repository to use
     * @param objectId   the identifier for a point in the repository. Can be a commit ID, branch or tag
     * @param path       the path to the file
     * @param supplier   an object that can return an OutputStream given a mime-type
     * @throws ServiceException if the stream operation fails.
     */
    void streamFile(@Nonnull Repository repository, @Nonnull String objectId, @Nonnull String path,
                    @Nonnull TypeAwareOutputSupplier supplier) throws ServiceException;
}
