package com.atlassian.bitbucket.content;

import com.atlassian.bitbucket.util.PagedCallback;

import javax.annotation.Nonnull;
import java.io.IOException;

/**
 * Describes a callback for receiving streaming {@link Change changes}.
 * <p>
 * Implementations of this callback performing internal paging may return {@code false} from {@link #onChange(Change)}
 * to indicate no more changes are desired. Such implementations are <i>encouraged</i> to also implement the
 * {@link PagedCallback PagedCallback} interface to allow the system to optimise output
 * handling, where possible, based on the page being requested.
 * <p>
 * Note: Implementors are <i>strongly</i> encouraged to extend from {@link AbstractChangeCallback}. This interface
 * <i>will</i> change, over time, and any class implementing it directly will be broken by such changes. Extending from
 * the abstract class will help prevent such breakages.
 *
 * @see AbstractChangeCallback
 */
@FunctionalInterface
public interface ChangeCallback {

    /**
     * Provides a {@link Change change} to the callback for processing.
     * <p>
     * If the callback is performing internal paging, it may signal the end of the page by returning {@code false} here.
     * When doing so, {@link #onEnd(ChangeSummary)} will never receive {@code true} for truncation.
     *
     * @param change the change to process
     * @return {@code true} if additional changes should be provided; otherwise, {@code false} if the callback has
     *         received as many changes as desired
     * @throws IOException may be thrown by implementations which perform I/O
     */
    boolean onChange(@Nonnull Change change) throws IOException;

    /**
     * Called after the final {@link #onChange(Change) change} has been streamed.
     * <p>
     * Note: If there were no changes, this method may be called immediately after {@link #onStart(ChangeContext)}
     * without any calls to {@link #onChange(Change)}.
     *
     * @param summary summarizes the request and the streamed changes
     * @throws IOException May be thrown by implementations which perform I/O.
     */
    default void onEnd(@Nonnull ChangeSummary summary) throws IOException {
    }

    /**
     * Called before the <i>first</i> {@link #onChange(Change) change} is streamed.
     *
     * @param context provides details about the request for which changes are being streamed
     * @throws IOException May be thrown by implementations which perform I/O.
     */
    default void onStart(@Nonnull ChangeContext context) throws IOException {
    }
}
