package com.atlassian.bitbucket.commit;

import com.atlassian.bitbucket.commit.graph.TraversalCallback;
import com.atlassian.bitbucket.commit.graph.TraversalRequest;
import com.atlassian.bitbucket.content.*;
import com.atlassian.bitbucket.idx.CommitIndex;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.scm.bulk.BulkCommitCallback;
import com.atlassian.bitbucket.util.Page;
import com.atlassian.bitbucket.util.PageRequest;
import com.atlassian.bitbucket.watcher.Watchable;
import com.atlassian.bitbucket.watcher.Watcher;

import javax.annotation.Nonnull;
import java.util.Optional;

public interface CommitService {

    /**
     * Retrieves a page of {@link Change changes} between two commits. If the {@link ChangesRequest#getSinceId() since}
     * commit is not specified, the {@link ChangesRequest#getUntilId() until} commit's <i>first</i> parent is used.
     * <p>
     * Note: The implementation will apply a hard cap ({@code page.max.changes}) and it is not
     * possible to request subsequent content when that cap is exceeded.
     *
     * @param request     describes the two commits to retrieve changes between and the repository which contains
     *                    <i>both commits</i>
     * @param pageRequest the start and limit for the page being requested
     * @return a page of changes within the repository between the specified IDs
     */
    @Nonnull
    Page<Change> getChanges(@Nonnull ChangesRequest request, @Nonnull PageRequest pageRequest);

    /**
     * Retrieves changesets for each of the requested {@link ChangesetsRequest#getCommitIds()} commits}, including the
     * first page of changes between each commit and its first parent.
     * <p>
     * If the {@link PageRequest#getLimit() page size} is less than the size of the collection of IDs, a subset of the
     * collection will be returned. Otherwise, changesets for all IDs in the requested collection are returned.
     *
     * @param request     request parameters, specifying the repository and the desired commits
     * @param pageRequest the start and limit of the page being request
     * @return a page containing changesets for a subset of the requested IDs
     * @throws NoSuchCommitException if {@link ChangesetsRequest#isIgnoreMissing() ignoreMissing} is not set and any
     *         of the requested commits cannot be found
     */
    @Nonnull
    Page<Changeset> getChangesets(@Nonnull ChangesetsRequest request, @Nonnull PageRequest pageRequest);

    /**
     * Retrieves a commit based on te provided {@link CommitRequest request}.
     * <p>
     * If only a {@link CommitRequest#getCommitId() commit ID} is specified, that commit will be returned if it exists.
     * If a {@link CommitRequest#getPath() path} is also provided, the first commit <i>starting from</i> the specified
     * commit ID which modifies that path is returned. If the specified commit does not exist, or if no commit in the
     * ancestry of that commit modifies the specified path, a {@link NoSuchCommitException} is thrown.
     * <p>
     * When retrieving a commit, {@link CommitRequest#getPropertyKeys() extra properties} can also be loaded.
     * Properties are metadata associated with commits by {@code CommitIndexer}s. Until 4.0, the extra properties
     * requested also control the attributes that are loaded. Attributes will be removed in 4.0.
     *
     * @param request describes the commit to retrieve, and the repository to retrieve it from
     * @return the first commit matching the request within the specified repository
     * @throws NoSuchCommitException if no matching commit can be found
     */
    @Nonnull
    Commit getCommit(@Nonnull CommitRequest request);

    /**
     * Retrieves a page of commits, starting from a given branch, tag or commit, optionally filtered to only return
     * commits which modify one or more paths. Commits are returned starting from (and including) the specified
     * {@link CommitsRequest#getCommitId()}, unless one or more {@link CommitsRequest#getPaths() paths} are provided.
     * Then the first commit returned, if any are returned, will be the first commit in the ancestry of the specified
     * starting commit which modifies <i>at least one of</i> the specified files.
     * <p>
     * This service call may be used to:
     * <ul>
     *     <li>Retrieve pages of commits on a given branch or tag</li>
     *     <li>Retrieve pages of ancestors starting from a given commit</li>
     *     <li>Retrieve pages of commits where at least one file from a specified set of files has been modified</li>
     * </ul>
     * Each request is for a <i>single starting point</i>. This means a single branch, tag or commit. For example,
     * retrieving all commits across all of a repository's branches where a given file has been modified in a single
     * request is not possible.
     * <p>
     * When retrieving commits, {@link CommitsRequest#getPropertyKeys() extra properties} can also be loaded.
     * Properties are metadata associated with commits by {@code CommitIndexer}s. Until 4.0, the extra properties
     * requested also control the attributes that are loaded. Attributes will be removed in 4.0.
     * <p>
     * Note: If any of the provided paths is renamed at some point in its history, commits will <i>not</i> follow the
     * path across the rename. In other words, only commits for exact paths provided will be included.
     *
     * @param request     describes the commits to retrieve, and the repository to retrieve them from
     * @param pageRequest the start and limit for the page being requested
     * @return a page of commits, which may be empty but will never be {@code null}
     */
    @Nonnull
    Page<Commit> getCommits(@Nonnull CommitsRequest request, @Nonnull PageRequest pageRequest);

    /**
     * Retrieves a page of commits between the {@link CommitsBetweenRequest#getIncludes() included commits} and the
     * {@link CommitsBetweenRequest#getExcludes() excluded commits}. Included and excluded commits may be specified
     * by branch or tag name or by commit ID. The commits returned are those which are reachable an included commit
     * and not reachable from any excluded commit. While this may imply an ancestral relationship between included
     * and excluded IDs, that is not strictly required. When the IDs are unrelated, all commits reachable from the
     * included commits are candidates for the returned page, and excluded commits are effectively ignored. One or
     * more paths may be provided to further narrow the results, restricting the returned commits to only which modify
     * <i>one or more</i> of the specified paths.
     * <p>
     * Which commits are included and which are excluded is important here. Consider the following example:
     * <ol>
     *     <li>Branch "feature-A" was created from "master" when "master" was at commit A</li>
     *     <li>Commits FA1, FA2 and FA3 were created on the "feature-A" branch</li>
     *     <li>Commits B and C were made on "master" after the branch was created</li>
     * </ol>
     * The graph would look like this:
     * <pre>
     * ---- A ---- B ---- C
     *       \
     *        \
     *        FA1 ---- FA2 ---- FA3
     * </pre>
     * If "master" is excluded and "feature-A" is included, commits FA1, FA2 and FA3 will be returned because they
     * are reachable from "feature-A" but not from "master". If the two IDs were reversed, commits B and C would be
     * returned, because they are reachable from "master" but not from "feature-A". This approach of swapping includes
     * and excludes can be used to determine how two branches have diverged over time.
     * <p>
     * When retrieving commits, {@link CommitRequest#getPropertyKeys() extra properties} can also be loaded.
     * Properties are metadata associated with commits by {@code CommitIndexer}s. Until 4.0, the extra properties
     * requested also control the attributes that are loaded. Attributes will be removed in 4.0.
     * <p>
     * <b>Warning:</b> Commits specified using branch or tag names will be resolved against the <b>primary</b>
     * {@link CommitsBetweenRequest#getRepository() repository}. When retrieving commits between repositories,
     * commits in the {@link CommitsBetweenRequest#getSecondaryRepository() secondary repository} <i>may only be
     * specified by commit ID</i>. This means requesting a page like this will not work:
     * <pre><code>
     *     CommitsBetweenRequest request = new CommitsBetweenRequest.Builder(repository)
     *             .exclude("master")
     *             .include("master")
     *             .secondaryRepository(forkOfRepository)
     *             .build();
     *     Page&lt;Commit&gt; page = commitService.getCommitsBetween(request, PageUtils.newRequest(0, 25));
     * </code></pre>
     * The returned page will always be empty because "master" was resolved on both sides using the same repository,
     * producing the same commit to include and exclude. This also demonstrates that <i>excludes take precedence</i>
     * over includes, as one might expect.
     *
     * @param request     describes the commits to include/exclude, paths to filter by and which repository, or
     *                    repositories, contain the commits
     * @param pageRequest the start and limit for the page being requested
     * @return a page containing 0 or more commits reachable from any included commit and not reachable from
     *         any excluded ones
     */
    @Nonnull
    Page<Commit> getCommitsBetween(@Nonnull CommitsBetweenRequest request, @Nonnull PageRequest pageRequest);

    /**
     * Retrieves the common ancestor for the provided commits
     *
     * @param request request describing which common ancestor to retrieve
     * @return the common ancestor of the provided commits, or {@link Optional#empty()} if the provided commits do
     *         not have a common ancestor
     * @since 5.0
     */
    @Nonnull
    Optional<MinimalCommit> getCommonAncestor(@Nonnull CommonAncestorRequest request);

    /**
     * Retrieve a {@link CommitDiscussion commit discussion} matching the provided request.
     *<p>
     * Note: Nullability actually depends on the provided request.
     * <ul>
     *     <li>If the discussion should be {@link CommitDiscussionRequest#isCreate() created}, the result will
     *     <i>never</i> be {@code null}</li>
     *     <li>Otherwise, if missing discussions should not be created, the result will be {@code null} if the
     *     requested {@link CommitDiscussionRequest#getCommitId() commit} has not been discussed previously</li>
     * </ul>
     *
     * @param request the request describing the discussion to retrieve or create
     * @return the retrieved discussion, which may be {@code null} depending on the request
     * @since 5.0
     */
    CommitDiscussion getDiscussion(@Nonnull CommitDiscussionRequest request);

    /**
     * Streams changes between the {@link ChangesRequest#getUntilId() until} and {@link ChangesRequest#getSinceId()
     * since} commits. If the {@link ChangesRequest#getSinceId() since} commit is not specified, the
     * {@link ChangesRequest#getUntilId() until} commit's <i>first</i> parent is used automatically.
     * <p>
     * Note: The implementation will apply a hard cap ({@code page.max.changes}) and it is not
     * possible to request subsequent content when that cap is exceeded.
     *
     * @param request  describes the two commits to retrieve changes between and the repository which contains
     *                 <i>both commits</i>
     * @param callback a callback to receive the changes
     */
    void streamChanges(@Nonnull ChangesRequest request, @Nonnull ChangeCallback callback);

    /**
     * Stream an arbitrary set of commits from an arbitrary number of repositories. The calling user is required
     * to have access to <i>all</i> of the {@link Repository repositories} from which commits are being requested
     * or an exception will be thrown.
     * <p>
     * Due to the bulk nature of this method, it is <i>required</i> that full {@link Commit#getId IDs} be used to
     * identify the commits to be streamed. <i>Short IDs, including {@link Commit#getDisplayId display IDs}, will
     * not work.</i> To load commits using a short ID, search first using the {@link CommitIndex}, which does accept
     * short IDs, and then use this method to resolve full {@link Commit}s for any matches the index returns.
     * <p>
     * Note: Commits may not be passed to the callback in the order they are requested.
     *
     * @param request the request, specifying which commits to retrieve and from which repositories
     * @param callback a callback to receive the commits
     * @since 5.8
     */
    void streamCommits(@Nonnull BulkCommitsRequest request, @Nonnull BulkCommitCallback callback);

    /**
     * Streams commits, starting from a given branch, tag or commit, optionally filtered to only include commits which
     * modify one or more paths. Commits are streamed starting from (and including) the specified
     * {@link CommitsRequest#getCommitId()}, unless one or more {@link CommitsRequest#getPaths() paths} are provided.
     * Then the first commit streamed, if any are, will be the first commit in the ancestry of the specified starting
     * commit which modifies <i>at least one of</i> the specified files.
     * <p>
     * This service call may be used to:
     * <ul>
     *     <li>Stream the commits on a given branch or tag</li>
     *     <li>Stream ancestors starting from a given commit</li>
     *     <li>Stream commits where at least one file from a specified set of files has been modified</li>
     * </ul>
     * Each request is for a <i>single starting point</i>. This means a single branch, tag or commit. For example,
     * streaming all commits across all of a repository's branches where a given file has been modified in a single
     * request is not possible.
     * <p>
     * When streaming commits, {@link CommitsRequest#getPropertyKeys() extra properties} can also be loaded.
     * Properties are metadata associated with commits by {@code CommitIndexer}s. Until 4.0, the extra properties
     * requested also control the attributes that are loaded. Attributes will be removed in 4.0.
     * <p>
     * Note: If any of the provided paths is renamed at some point in its history, commits will <i>not</i> follow the
     * path across the rename. In other words, only commits for exact paths provided will be included.
     *
     * @param request  describes the commits to stream, and the repository to stream them from
     * @param callback a callback to receive the commits
     */
    void streamCommits(@Nonnull CommitsRequest request, @Nonnull CommitCallback callback);

    /**
     * Streams commits between the {@link CommitsBetweenRequest#getIncludes() included commits} and the
     * {@link CommitsBetweenRequest#getExcludes() excluded commits}.
     * <p>
     * This method promises to call {@link CommitCallback#onStart(CommitContext)} on the supplied callback exactly once,
     * followed by zero or more calls to {@link CommitCallback#onCommit(Commit)}, one for each commit to be streamed
     * (until {@link CommitCallback#onCommit(Commit) onCommit(...)} returns {@code false} or there are no more commits),
     * followed by exactly one call to {@link CommitCallback#onEnd(CommitSummary)}.
     * <p>
     * See {@link #getCommitsBetween(CommitsBetweenRequest, PageRequest) getCommitsBetween(...)} for a
     * description of the semantics of the commits returned.
     *
     * @param request  describes the commits to include/exclude, paths to filter by and which repository, or
     *                 repositories, contain the commits
     * @param callback a callback to receive the commits
     */
    void streamCommitsBetween(@Nonnull CommitsBetweenRequest request, @Nonnull CommitCallback callback);

    /**
     * Streams diff output for the specified {@link DiffRequest#getPaths() paths} at the specified commit.
     * <p>
     * Note: This interface is currently <i>not paged</i>. The implementation will apply a hard cap
     * ({@code page.max.diff.lines}) and it is not possible to request subsequent content when that cap is exceeded.
     *
     * @param request  the repository, starting and terminating commits, and paths to filter by
     * @param callback a callback to receive the diff details
     */
    void streamDiff(@Nonnull DiffRequest request, @Nonnull DiffContentCallback callback);

    /**
     * Streams the last modification for files under the requested {@link LastModifiedRequest#getPath path}, starting
     * from the specified {@link LastModifiedRequest#getCommitId() commit}.
     * <p>
     * The {@link LastModifiedRequest#getCommitId commit ID} serves as a <i>starting point</i>. For any files under
     * the requested path, commits are traversed from that starting point until the most recent commit to <i>directly
     * modify</i> each file is found. Because the exact number of commits which must be traversed is unknown, this
     * operation may be very expensive.
     *
     * @param request  the repository, path and starting commit
     * @param callback a callback to receive last modification data
     * @since 4.6
     */
    void streamLastModified(@Nonnull LastModifiedRequest request, @Nonnull LastModifiedCallback callback);

    /**
     * Traverse the graph of commits in topological order, optionally supplying included commits to start from
     * and excluded commits to ignore.
     *
     * @param request  the traversal request describing the repository and the commits to include and exclude
     * @param callback a callback to receive the traversed commits
     */
    void traverse(@Nonnull TraversalRequest request, @Nonnull TraversalCallback callback);

    /**
     * Removes the current user as a watcher for the specified commit, which will prevent further notifications from
     * being dispatched to that user.
     *
     * @param repository the ID of the repository containing the commit
     * @param commitId   the ID of the commit within the repository
     * @return {@code true} if the watcher existed (and thus was removed); otherwise {@code false}
     * @deprecated in 5.10 for removal in 6.0. Use {@link com.atlassian.bitbucket.watcher.WatcherService#unwatch(Watchable)} instead
     */
    @Deprecated
    boolean unwatch(@Nonnull Repository repository, @Nonnull String commitId);

    /**
     * Adds the current user as a watcher for the specified commit. Watchers receive notifications when new comments
     * are added to the commit.
     *
     * @param repository the ID of the repository containing the commit
     * @param commitId   the ID of the commit within the repository
     * @return the created watcher
     * @deprecated in 5.10 for removal in 6.0. Use {@link com.atlassian.bitbucket.watcher.WatcherService#watch(Watchable)} instead
     */
    @Nonnull
    @Deprecated
    Watcher watch(@Nonnull Repository repository, @Nonnull String commitId);
}
