package com.atlassian.bitbucket.commit;

import com.atlassian.bitbucket.repository.Branch;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.repository.Tag;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Set;

/**
 * Describes a request for a single commit. If no {@link #getPath path} is provided, the specified {@link #getCommitId
 * commit} will be the one returned. If a {@link #getPath path} <i>is</i> provided, the commit returned will be the
 * first commit in the <i>ancestry</i> of the specified {@link #getCommitId commit} which modifies that path.
 */
//TODO in 5.0: Remove methods overridden from AbstractCommitRequest
public class CommitRequest extends AbstractCommitRequest {

    private final Set<String> propertyKeys;

    private CommitRequest(Builder builder) {
        super(builder);

        propertyKeys = builder.propertyKeys.build();
    }

    /**
     * @return the {@link Commit#getId commit ID}, which may also be specified using a {@link Branch#getId branch ID}
     *         or {@link Tag#getId() tag ID}
     */
    @Nonnull
    @Override
    public String getCommitId() {
        return super.getCommitId();
    }

    /**
     * @return the max message length to be processed for any of the commits, which may be
     *         {@link #UNLIMITED_MESSAGE_LENGTH} to include the full commit message regardless of length
     * @since 4.5
     */
    @Override
    public int getMaxMessageLength() {
        return super.getMaxMessageLength();
    }

    /**
     * Retrieves an optional path which is used to control which commit is returned. When a path is provided, the
     * {@link #getCommitId() commit ID} is used as a <i>starting point</i> and may not actually be the commit that
     * ultimately gets returned. Instead, the first commit which modifies the specified path will be returned. If
     * none of the commits in the ancestry of the starting commit modify the specified path, no commit is returned.
     *
     * @return an optional path
     */
    @Nullable
    @Override
    public String getPath() {
        return super.getPath();
    }

    /**
     * @return a set of {@link Commit#getProperties() property keys} to load, which may be empty but never {@code null}
     */
    @Nonnull
    public Set<String> getPropertyKeys() {
        return propertyKeys;
    }

    /**
     * @return the repository the commit should be loaded from
     */
    @Nonnull
    @Override
    public Repository getRepository() {
        return super.getRepository();
    }

    public static class Builder extends AbstractCommitRequestBuilder<Builder> {

        private final ImmutableSet.Builder<String> propertyKeys;

        public Builder(@Nonnull Repository repository, @Nonnull String commitId) {
            super(repository, commitId);

            propertyKeys = ImmutableSet.builder();
        }

        @Nonnull
        public CommitRequest build() {
            return new CommitRequest(this);
        }

        /**
         * @since 4.5
         */
        @Nonnull
        @Override
        public Builder maxMessageLength(int value) {
            return super.maxMessageLength(value);
        }

        @Nonnull
        @Override
        public Builder path(@Nullable String value) {
            return super.path(value);
        }

        @Nonnull
        public Builder propertyKey(@Nullable String value) {
            addIf(NOT_BLANK, propertyKeys, value);

            return self();
        }

        @Nonnull
        public Builder propertyKeys(@Nullable String value, @Nullable String... values) {
            addIf(NOT_BLANK, propertyKeys, value, values);

            return self();
        }

        @Nonnull
        public Builder propertyKeys(@Nullable Iterable<String> values) {
            addIf(NOT_BLANK, propertyKeys, values);

            return self();
        }

        @Override
        protected Builder self() {
            return this;
        }
    }
}
