package com.atlassian.bitbucket.comment;

import com.atlassian.bitbucket.util.Page;
import com.atlassian.bitbucket.util.PageRequest;

import javax.annotation.Nonnull;
import java.util.Optional;

/**
 * @since 5.0
 */
public interface CommentService {

    /**
     * Adds a general comment on a commentable
     *
     * @param request describes the comment to create
     * @return the created comment
     */
    @Nonnull
    Comment addComment(@Nonnull AddCommentRequest request);

    /**
     * Adds a comment on a file in a diff
     *
     * @param request describes the comment to create
     * @return the created comment
     */
    @Nonnull
    Comment addComment(@Nonnull AddFileCommentRequest request);

    /**
     * Adds a comment on a line on a file in a diff
     *
     * @param request describes the comment to create
     * @return the created comment
     */
    @Nonnull
    Comment addComment(@Nonnull AddLineCommentRequest request);

    /**
     * Adds a comment reply to a {@link CommentThread tread}
     *
     * @param request a request describing the comment to be added
     * @return the created comment
     * @throws NoSuchCommentException if the parent comment cannot be found
     */
    @Nonnull
    Comment addReply(@Nonnull AddCommentReplyRequest request);

    /**
     * Count all the {@link Comment comments} matching the provided request.
     * <p>
     * If present, each one of the properties in the search request applies an extra filter to the search
     * criteria. <strong>A {@code null} value (or an empty collection) means no filter will be applied
     * for that property</strong>. {@link CommentThreadDiffAnchor#isOrphaned() Orphaned} comments are not
     * counted by this endpoint.
     *
     * @param request the request describing the search
     * @return the number of comments matching the request
     */
    long countComments(@Nonnull CommentSearchRequest request);

    /**
     * Count all the {@link CommentThread threads} matching the provided request.
     * <p>
     * If present, each one of the properties in the search request applies an extra filter to the search
     * criteria. <strong>A {@code null} value (or an empty collection) means no filter will be applied
     * for that property</strong>. {@link CommentThreadDiffAnchor#isOrphaned() Orphaned} comments are not
     * counted by this endpoint.
     *
     *
     * @param request the request describing the search
     * @return the number of threads matching the request
     */
    long countThreads(@Nonnull CommentSearchRequest request);

    /**
     * Delete a comment
     *
     * @param commentId the ID for the comment that should be deleted
     * @param version the expected version for the comment to delete
     * @return {@code true} if the comment existed and was deleted, {@code false} otherwise
     * @throws CommentDeletionException if the comment has replies
     * @throws CommentOutOfDateException if the provided {@code version} doesn't match the comment's current version
     * @throws NoSuchCommentException if the comment to delete cannot be found
     */
    boolean deleteComment(long commentId, int version);

    /**
     * Find a comment with the given ID
     *
     * @param commentId the ID for the comment to look for
     * @return the comment with the given ID
     */
    @Nonnull
    Optional<Comment> getComment(long commentId);

    /**
     * Search for all {@link CommentThread threads} matching the provided request.
     * <p>
     * If present, each one of the properties in the search request applies an extra filter to the search
     * criteria. <strong>A {@code null} value (or an empty collection) means no filter will be applied
     * for that property</strong>. {@link CommentThreadDiffAnchor#isOrphaned() Orphaned} comments are not
     * returned by this endpoint.
     *
     * @param request the request describing the search
     * @param pageRequest the request specifying the start and limit of the page
     * @return the threads matching the request
     */
    @Nonnull
    Page<CommentThread> searchThreads(@Nonnull CommentSearchRequest request, @Nonnull PageRequest pageRequest);

    /**
     * Updates a {@link Comment comment} according to the provided {@link CommentUpdateRequest request}.
     *
     * @param request the request describing how the comment should be updated
     * @return the updated comment
     * @throws CommentOutOfDateException if the provided {@code version} doesn't match the comment's current version
     * @throws NoSuchCommentException if the comment to update cannot be found
     */
    @Nonnull
    Comment updateComment(@Nonnull CommentUpdateRequest request);
}
