package com.atlassian.bitbucket.comment;

import com.atlassian.bitbucket.util.BuilderSupport;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Objects;
import java.util.Set;

import static com.atlassian.bitbucket.comment.CommentThreadDiffAnchorState.ACTIVE;
import static java.util.Objects.requireNonNull;

/**
 * A request for searching for {@link Comment comments} and {@link CommentThread threads}.
 *
 * @since 5.0
 */
public class CommentSearchRequest {

    private final CommentThreadDiffAnchorState anchorState;
    private final Commentable commentable;
    private final Set<CommentThreadDiffAnchorType> diffTypes;
    private final String fromHash;
    private final String path;
    private final String toHash;

    private CommentSearchRequest(@Nonnull Builder builder) {
        anchorState = builder.anchorState;
        commentable = builder.commentable;
        diffTypes = builder.diffTypes.build();
        fromHash = builder.fromHash;
        path = builder.path;
        toHash = builder.toHash;
    }

    /**
     * @since 5.4
     */
    @Nonnull
    public CommentThreadDiffAnchorState getAnchorState() {
        return anchorState;
    }

    @Nonnull
    public Commentable getCommentable() {
        return commentable;
    }

    @Nonnull
    public Set<CommentThreadDiffAnchorType> getDiffTypes() {
        return diffTypes;
    }

    @Nullable
    public String getFromHash() {
        return fromHash;
    }

    @Nullable
    public String getPath() {
        return path;
    }

    @Nullable
    public String getToHash() {
        return toHash;
    }

    public static class Builder extends BuilderSupport {

        private final Commentable commentable;
        private final ImmutableSet.Builder<CommentThreadDiffAnchorType> diffTypes;

        private CommentThreadDiffAnchorState anchorState;
        private String fromHash;
        private String path;
        private String toHash;

        public Builder(@Nonnull Commentable commentable) {
            anchorState = ACTIVE;
            this.commentable = requireNonNull(commentable, "commentable");
            diffTypes = ImmutableSet.builder();
        }

        /**
         * @since 5.4
         */
        @Nonnull
        public Builder anchorState(@Nonnull CommentThreadDiffAnchorState value) {
            anchorState = requireNonNull(value, "anchorState");
            return this;
        }

        @Nonnull
        public CommentSearchRequest build() {
            return new CommentSearchRequest(this);
        }

        @Nonnull
        public Builder diffType(@Nullable CommentThreadDiffAnchorType value) {
            addIf(Objects::nonNull, diffTypes, value);
            return this;
        }

        @Nonnull
        public Builder diffTypes(@Nullable Iterable<CommentThreadDiffAnchorType> values) {
            addIf(Objects::nonNull, diffTypes, values);
            return this;
        }

        @Nonnull
        public Builder fromHash(@Nullable String value) {
            fromHash = value;
            return this;
        }

        @Nonnull
        public Builder path(@Nullable String value) {
            path = value;
            return this;
        }

        @Nonnull
        public Builder toHash(@Nullable String value) {
            toHash = value;
            return this;
        }
    }
}
