package com.atlassian.bitbucket.avatar;

import com.atlassian.bitbucket.project.Project;
import com.atlassian.bitbucket.user.Person;

import javax.annotation.Nonnull;

/**
 * Provides URLs from which avatar images can be retrieved.
 */
public interface AvatarService {

    /**
     * Retrieves the maximum size, in bytes, for uploaded avatars. Avatars larger than this will be rejected by
     * the server.
     *
     * @return the maximum size, in bytes, for uploaded avatars
     * @since 4.13
     */
    long getMaxUploadSize();

    /**
     * Retrieves a URL referencing an avatar for the provided {@link Project project}.
     * <p>
     * Implementations of this interface <i>shall not</i> return {@code null}. If no avatar is available for the
     * provided {@code project}, the URL of a default avatar is returned. The default avatar for a project will
     * be randomly assigned, but will be stable once set.
     *
     * @param project the project whose avatar is being requested
     * @param request a request describing the avatar being requested
     * @return a URL referencing an avatar for the provided {@code project}
     */
    @Nonnull
    String getUrlForProject(@Nonnull Project project, @Nonnull AvatarRequest request);

    /**
     * Retrieves a URL referencing an avatar for the provided {@link Person person}.
     * <p>
     * Implementations of this interface <i>shall not</i> return {@code null}. If no avatar is available for the
     * provided {@code person}, the URL of a default avatar is returned.
     *
     * @param person  the person whose avatar is being requested
     * @param request a request describing the avatar being requested
     * @return a URL referencing an avatar for the provided {@code person}
     */
    @Nonnull
    String getUrlForPerson(@Nonnull Person person, @Nonnull AvatarRequest request);
}
