define('hipchat/feature/room-selector', [
    'jquery',
    'lodash',
    'bitbucket/util/server',
    'exports'
], function (
    $,
    _,
    server,
    exports
) {
    'use strict';

    var CONTAINER_CSS_CLASS = 'select2-hipchat-room-select';
    var ROOM_ENDPOINT = '/rest/hipchat/integration/1.0/rooms';

    var roomListPromise = null;

    /**
     * Gets the room list
     *
     * @returns {Promise} Promise that will resolve wih the room data.
     */
    function getRoomListPromise () {
        if (roomListPromise === null) {
            roomListPromise = server.rest({
                url: AJS.contextPath() + ROOM_ENDPOINT,
                cache: false
            });
        }
        return roomListPromise;
    }

    /**
     * Filters the room list and updates the {@link query}.
     *
     * @param {Promise} roomListPromise a promise that will resolve to the full room list.
     * @param {Function<Room>} filterFunc should return true if the passed in room should be shown.
     * @param {Function} query
     */
    function roomQuery (roomListPromise, filterFunc, query) {
        roomListPromise.done(function (roomList) {
            var rooms = roomList;
            var trimmedSearchText = query.term.trim();
            var searchStrLower = trimmedSearchText.toLocaleLowerCase();
            var filteredRooms = _.filter(rooms, function (item) {
                item.text = item.name;
                return item.name.toLocaleLowerCase().indexOf(searchStrLower) >= 0 && filterFunc(item);
            });

            var results = [];
            var matchCount = filteredRooms.length;
            if (matchCount >= 1) {
                results.push({
                    text: AJS.I18n.getText('bitbucket.plugins.hipchat.room-selector.title'),
                    children: filteredRooms
                });
            }
            if (!HipChat.UserLink.context.userLinked) {
                results.push({
                    text: AJS.I18n.getText('bitbucket.plugins.hipchat.room-selector.private'),
                    children: [{
                        isHipchatUserLink: true,
                        id: 'hipchat-user-link', // needs an id to be selectable
                        text: AJS.I18n.getText('bitbucket.plugins.hipchat.room-selector.login')
                    }]
                });
            }
            query.callback({ results: results });
        });
    }

    /**
     * Renders a room item
     *
     * @param {Room} room - the room item to render
     * @returns {string} the rendered HTML.
     */
    var roomRenderer = function (room) {
        if (room.isHipchatUserLink) {
            return hipchat.integration.plugin.roomSelector.renderLoginLink();
        }
        if (!room.id) {
            return room.text;
        }
        return hipchat.integration.plugin.roomSelector.renderRoomItem({
            id: room.id,
            name: room.name,
            privateRoom: room['private']
        });
    };

    /**
     * Creates a room selector.
     * @param {jQuery} $el - jQuery element to create the select2 out of.
     * @param {Function<Room>} filterFunc - Should return true if the passed in room should be included in the room list.
     * @returns {{getSelection: Function, select2: auiSelect2}} An object for controlling the select2.
     */
    exports.createRoomSelect = function ($el, filterFunc) {
        var roomListPromise = getRoomListPromise();

        var roomSelect = $el.auiSelect2({
            placeholder: AJS.I18n.getText('bitbucket.plugins.hipchat.room-selector.placeholder'),
            containerCssClass: CONTAINER_CSS_CLASS,
            dropdownCssClass: 'select2-hipchat-room-dropdown',
            query: roomQuery.bind(null, roomListPromise, filterFunc),
            allowClear: true,
            formatResult: roomRenderer,
            formatSelection: roomRenderer
        });

        var selectedRoom;

        roomSelect.on('select2-selected', function (event) {
            if (event.choice.id === 'hipchat-user-link') {
                HipChat.UserLink.linkUser(event);
            }
            selectedRoom = event.choice;
        });

        var $roomSearchInput = $('.' + CONTAINER_CSS_CLASS + ' .select2-input');
        if ($roomSearchInput.length && "placeholder" in $roomSearchInput[0]) {
            $roomSearchInput.attr("placeholder", AJS.I18n.getText('bitbucket.plugins.hipchat.room-selector.search'));
        }

        return {
            getSelection: function () {
                return selectedRoom;
            },
            select2: roomSelect
        }
    };


});