/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.beehive.db;

import com.atlassian.beehive.core.ClusterLockStatus;
import com.atlassian.beehive.core.ManagedClusterLock;
import com.atlassian.beehive.core.stats.StatisticsKey;
import com.atlassian.beehive.db.DatabaseClusterLockLeaseRenewer;
import com.atlassian.beehive.db.DeadOwnerThreadException;
import com.atlassian.beehive.db.StatisticsHolder;
import com.atlassian.beehive.db.spi.ClusterLockDao;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import java.lang.ref.WeakReference;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Condition;
import java.util.function.BooleanSupplier;
import javax.annotation.Nonnull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class DatabaseClusterLock
implements ManagedClusterLock {
    private static final int INITIAL_SLEEP_MILLIS = 100;
    private static final int MAX_SLEEP_MILLIS = 10000;
    private static final Logger log = LoggerFactory.getLogger(DatabaseClusterLock.class);
    private static final int MAX_RETRIES = 3;
    private final String lockName;
    private final ClusterLockDao clusterLockDao;
    private final AtomicReference<Owner> ownerRef = new AtomicReference();
    private final AtomicInteger depth = new AtomicInteger();
    private final StatisticsHolder stats = new StatisticsHolder();
    private final Supplier<ClusterLockStatus> databaseLockStatusSupplier;
    private final AtomicBoolean inserted = new AtomicBoolean(false);
    private final Object monitor = new Object();
    private DatabaseClusterLockLeaseRenewer databaseClusterLockLeaseRenewer;

    public DatabaseClusterLock(String lockName, ClusterLockDao clusterLockDao, DatabaseClusterLockLeaseRenewer databaseClusterLockLeaseRenewer) {
        this.lockName = lockName;
        this.clusterLockDao = clusterLockDao;
        this.databaseClusterLockLeaseRenewer = databaseClusterLockLeaseRenewer;
        this.databaseLockStatusSupplier = Suppliers.memoizeWithExpiration(() -> {
            ClusterLockStatus clusterLockStatus = clusterLockDao.getClusterLockStatusByName(lockName);
            if (clusterLockStatus == null) {
                clusterLockDao.insertEmptyClusterLock(lockName);
                return clusterLockDao.getClusterLockStatusByName(lockName);
            }
            return clusterLockStatus;
        }, (long)100L, (TimeUnit)TimeUnit.MILLISECONDS);
    }

    @Nonnull
    public String getName() {
        return this.lockName;
    }

    public boolean isLocked() {
        return this.getClusterLockStatus().getLockedByNode() != null;
    }

    boolean isLockedLocally() {
        Owner owner = this.ownerRef.get();
        return owner != null && owner.getThread() != null && owner.getThread().isAlive();
    }

    public void interruptOwner() {
        Thread ownerThread;
        Owner owner = this.ownerRef.get();
        if (owner != null && (ownerThread = owner.getThread()) != null) {
            ownerThread.interrupt();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void lock() {
        long startedAt = this.nowInMillis();
        boolean wasInterrupted = Thread.interrupted();
        if (!this.tryLock()) {
            this.stats.tallyWaitBegin();
            try {
                this.uninterruptibleWait();
            }
            finally {
                this.stats.tallyWaitEndAfter(this.nowInMillis() - startedAt);
            }
        }
        DatabaseClusterLock.interruptIf(wasInterrupted);
    }

    private void uninterruptibleWait() {
        boolean wasInterrupted = false;
        int sleepTimeMillis = 100;
        do {
            try {
                this.sleep(sleepTimeMillis);
            }
            catch (InterruptedException ie) {
                wasInterrupted = true;
            }
            sleepTimeMillis = Math.min(sleepTimeMillis * 2, 10000);
        } while (!this.tryLock());
        DatabaseClusterLock.interruptIf(wasInterrupted);
    }

    public void lockInterruptibly() throws InterruptedException {
        long startedAt = this.nowInMillis();
        if (Thread.interrupted()) {
            throw new InterruptedException();
        }
        if (this.tryLock()) {
            return;
        }
        this.stats.tallyWaitBegin();
        try {
            this.interruptibleWait();
        }
        finally {
            this.stats.tallyWaitEndAfter(this.nowInMillis() - startedAt);
        }
    }

    private void interruptibleWait() throws InterruptedException {
        int sleepTimeMillis = 100;
        do {
            this.sleep(sleepTimeMillis);
            sleepTimeMillis = Math.min(sleepTimeMillis * 2, 10000);
        } while (!this.tryLock());
    }

    public boolean tryLock() {
        try {
            log.trace("Attempt to get cluster lock '{}' by {}.", (Object)this.lockName, (Object)Thread.currentThread());
            if (this.tryLockLocally()) {
                return Attempt.doTry(this::tryLockRemotely).withOnFalse(this::releaseThreadLock).withOnException(this::releaseThreadLock).go();
            }
            Owner owner = this.ownerRef.get();
            if (this.tryReenterLock(owner)) {
                return true;
            }
            if (this.tryEvictDeadOwnerThread(owner)) {
                return Attempt.doTry(this::tryLockRemotely).withOnFalse(this::releaseThreadLock).withOnException(this::releaseThreadLock).go();
            }
            log.debug("Acquisition of cluster lock '{}' by {} failed. Lock is owned this node's '{}' ", new Object[]{this.lockName, Thread.currentThread(), owner.getThread()});
            this.stats.tallyFailLocal();
            return false;
        }
        catch (IllegalMonitorStateException t) {
            this.stats.tallyStateError(this.nowInMillis());
            throw t;
        }
        catch (Throwable t) {
            this.stats.tallyError(this.nowInMillis());
            throw t;
        }
    }

    private boolean tryLockRemotely() {
        this.tryInsert();
        if (this.clusterLockDao.tryAcquireLock(this.lockName)) {
            this.databaseClusterLockLeaseRenewer.onLock(this);
            log.debug("Cluster lock '{}' was successfully acquired by this node's {}.", (Object)this.lockName, (Object)Thread.currentThread());
            this.depth.set(1);
            this.stats.tallyLockedAt(this.nowInMillis());
            return true;
        }
        log.debug("Acquisition of cluster lock '{}' by {} failed. Lock is owned by another node.", (Object)this.lockName, (Object)Thread.currentThread());
        this.stats.tallyFailRemote(this.nowInMillis());
        return false;
    }

    private void tryInsert() {
        if (!this.inserted.get()) {
            this.clusterLockDao.insertEmptyClusterLock(this.lockName);
            this.inserted.compareAndSet(false, true);
        }
    }

    private boolean tryLockLocally() {
        return this.ownerRef.compareAndSet(null, new Owner(Thread.currentThread()));
    }

    private boolean tryReenterLock(Owner owner) {
        Thread me = Thread.currentThread();
        if (owner.getThread() != me) {
            return false;
        }
        int currentDepth = this.depth.incrementAndGet();
        log.trace("Cluster lock '{}' was successfully reentered by '{}', depth increased to {}", new Object[]{this.lockName, me, currentDepth});
        if (currentDepth < 0) {
            this.depth.decrementAndGet();
            throw new IllegalMonitorStateException("Maximum lock count exceeded");
        }
        return true;
    }

    private boolean isLockedByDeadThread(Owner owner) {
        Thread ownerThread = owner.getThread();
        if (ownerThread == null || !ownerThread.isAlive()) {
            return true;
        }
        log.debug("Cluster lock '{}' currently held by another local thread '{}'.", (Object)this.lockName, (Object)ownerThread.getName());
        return false;
    }

    private boolean tryEvictDeadOwnerThread(Owner owner) {
        Thread me;
        if (this.isLockedByDeadThread(owner) && this.ownerRef.compareAndSet(owner, new Owner(me = Thread.currentThread()))) {
            log.error("During attempt to acquire lock '{}' by '{}' ownership by dead thread was detected. '{}' terminated before unlocking. Evicting previous owner and attempting to confirm ownership in DB...", new Object[]{this.lockName, me, owner});
            this.stats.tallyForcedUnlock();
            return true;
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void unlock() {
        Thread me = Thread.currentThread();
        this.assertLocallyOwned(me);
        if (!this.tryExitReenteredLock()) {
            Object object = this.monitor;
            synchronized (object) {
                log.trace("Cluster lock '{}' attempting unlock by '{}'", (Object)this.lockName, (Object)me);
                boolean success = false;
                int retries = 0;
                try {
                    do {
                        try {
                            this.databaseClusterLockLeaseRenewer.onUnlock(this);
                            this.clusterLockDao.unlock(this.lockName);
                            success = true;
                            log.debug("Cluster lock '{}' unlocked by '{}'", (Object)this.lockName, (Object)me);
                        }
                        catch (IllegalMonitorStateException ex) {
                            String errorReport = StatisticsHolder.getStatisticsSummary(this.getStatistics(), this.nowInMillis());
                            log.error("Cluster lock '{}' by '{}' unlock failed, held by someone else. " + errorReport, (Object)this.lockName, (Object)me);
                            this.stats.tallyStateError(this.nowInMillis());
                            break;
                        }
                        catch (Exception ex) {
                            if (++retries > 3) {
                                String errorReport = StatisticsHolder.getStatisticsSummary(this.getStatistics(), this.nowInMillis());
                                log.error("Unable to unlock " + this.toString() + ", Number of retries exceeded, rethrowing ." + errorReport, (Throwable)ex);
                                this.stats.tallyError(this.nowInMillis());
                                throw ex;
                            }
                            log.error("Unable to unlock " + this.toString() + ", retrying.", (Throwable)ex);
                            this.stats.tallyError(this.nowInMillis());
                            try {
                                this.sleep(100L);
                            }
                            catch (InterruptedException ie) {
                                Thread.currentThread().interrupt();
                            }
                        }
                    } while (!success && !Thread.currentThread().isInterrupted());
                }
                finally {
                    this.unlockLocally();
                }
            }
        }
    }

    boolean renew() {
        Object object = this.monitor;
        synchronized (object) {
            if (Thread.currentThread().isInterrupted()) {
                return false;
            }
            if (this.isLockedLocally()) {
                boolean b = this.tryRenew();
                return b;
            }
            throw new DeadOwnerThreadException();
        }
    }

    private boolean tryRenew() {
        try {
            this.renewInDB();
            return true;
        }
        catch (IllegalMonitorStateException ex) {
            this.reportRenewStateError(ex);
            this.interruptOwner();
            return this.tryReacquireRemoteLockSafely();
        }
        catch (Throwable t) {
            this.reportRenewGeneralError(t);
            return false;
        }
    }

    private void reportRenewGeneralError(Throwable t) {
        String errorReport = StatisticsHolder.getStatisticsSummary(this.getStatistics(), this.nowInMillis());
        log.error("Failed to renew lease on lock: , " + this.getName() + ". Error occured during attempt to renew this lock in the database. Will retry on next scheduled run..." + errorReport, t);
        this.stats.tallyError(this.nowInMillis());
    }

    private void reportRenewStateError(IllegalMonitorStateException ex) {
        String errorReport = StatisticsHolder.getStatisticsSummary(this.getStatistics(), this.nowInMillis());
        log.error("Failed to renew lease on lock: " + this.getName() + ". this lock is no longer owned by this node in the database, attempting to reacquire it... " + errorReport, (Throwable)ex);
        this.stats.tallyStateError(this.nowInMillis());
    }

    private boolean tryReacquireRemoteLockSafely() {
        try {
            if (this.clusterLockDao.tryAcquireLock(this.getName())) {
                log.warn("Successfully reacquired lock: " + this.getName() + " after it was lost it for some time.");
                return true;
            }
            log.error("Failed to reacquire lock: " + this.getName() + ". Will retry until success or local unlock on next scheduled renewer runs...");
            return false;
        }
        catch (Throwable e) {
            log.error("Error during attempt to reacquire lock: " + this.getName() + ". Will retry until success or local unlock on next scheduled renewer runs...", e);
            return false;
        }
    }

    private void renewInDB() {
        this.clusterLockDao.renewLease(this.getName());
        this.stats.tallyRenewed(this.nowInMillis());
    }

    private boolean tryExitReenteredLock() {
        int currentDepth = this.depth.decrementAndGet();
        if (currentDepth > 0) {
            log.trace("Reentered Cluster lock '{}' depth decremented by '{}' to {}", new Object[]{this.lockName, Thread.currentThread(), currentDepth});
            return true;
        }
        return false;
    }

    private void assertLocallyOwned(Thread me) {
        Owner owner = this.ownerRef.get();
        if (owner == null || owner.getThread() != me) {
            throw new IllegalMonitorStateException("Cluster lock '" + this.lockName + "' cannot be unlocked because it is not owned by this thread: " + me + " (owner: " + (owner == null || owner.getThread() == null ? "null" : owner.getThread().getName()) + ")");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean tryLock(long waitTime, @Nonnull TimeUnit unit) throws InterruptedException {
        if (Thread.interrupted()) {
            throw new InterruptedException();
        }
        long startedAt = this.nowInMillis();
        if (this.tryLock()) {
            return true;
        }
        long deadline = startedAt + unit.toMillis(waitTime);
        this.stats.tallyWaitBegin();
        try {
            boolean bl = this.tryLockWaitWithTimeout(deadline);
            return bl;
        }
        finally {
            this.stats.tallyWaitEndAfter(this.nowInMillis() - startedAt);
        }
    }

    private boolean tryLockWaitWithTimeout(long deadline) throws InterruptedException {
        long sleepTimeMillis = 100L;
        do {
            long remainingWaitTime;
            if ((remainingWaitTime = deadline - this.nowInMillis()) <= 0L) {
                return false;
            }
            sleepTimeMillis = Math.min(sleepTimeMillis, remainingWaitTime);
            this.sleep(sleepTimeMillis);
            sleepTimeMillis = Math.min(sleepTimeMillis * 2L, 10000L);
        } while (!this.tryLock());
        return true;
    }

    private void releaseThreadLock() {
        Owner currentOwner = this.ownerRef.get();
        if (Thread.currentThread().equals(currentOwner.getThread())) {
            this.ownerRef.compareAndSet(currentOwner, null);
        }
    }

    private void unlockLocally() {
        this.releaseThreadLock();
        this.stats.tallyUnlockedAt(this.nowInMillis());
    }

    @VisibleForTesting
    long nowInMillis() {
        return System.currentTimeMillis();
    }

    public boolean isHeldByCurrentThread() {
        Owner owner = this.ownerRef.get();
        return owner != null && owner.getThread() == Thread.currentThread();
    }

    @Nonnull
    public Condition newCondition() {
        throw new UnsupportedOperationException("newCondition() not supported in ClusterLock");
    }

    @Nonnull
    public ClusterLockStatus getClusterLockStatus() {
        ClusterLockStatus lock = (ClusterLockStatus)this.databaseLockStatusSupplier.get();
        return lock;
    }

    @Nonnull
    public Map<StatisticsKey, Long> getStatistics() {
        return this.stats.getStatistics();
    }

    @VisibleForTesting
    void sleep(long timeout) throws InterruptedException {
        Thread.sleep(timeout);
    }

    private static void interruptIf(boolean wasInterrupted) {
        if (wasInterrupted) {
            Thread.currentThread().interrupt();
        }
    }

    static class Owner {
        private final WeakReference<Thread> thd;
        private final String name;

        Owner(Thread me) {
            this.thd = new WeakReference<Thread>(me);
            this.name = me.getName();
        }

        Thread getThread() {
            return (Thread)this.thd.get();
        }

        public String toString() {
            return this.name;
        }
    }

    private static class Attempt {
        private Optional<Runnable> onFalse = Optional.empty();
        private Optional<Runnable> onException = Optional.empty();
        private BooleanSupplier action;

        private Attempt(Attempt builder) {
            this.onFalse = builder.onFalse;
            this.onException = builder.onException;
            this.action = builder.action;
        }

        private Attempt(BooleanSupplier action) {
            this.action = action;
        }

        public static Attempt doTry(BooleanSupplier action) {
            return new Attempt(action);
        }

        private boolean perform() {
            try {
                boolean result = this.action.getAsBoolean();
                if (!result) {
                    this.onFalse.ifPresent(Runnable::run);
                    return false;
                }
                return result;
            }
            catch (Exception ex) {
                this.onException.ifPresent(Runnable::run);
                throw ex;
            }
        }

        public Attempt withOnFalse(Runnable onFailure) {
            this.onFalse = Optional.of(onFailure);
            return this;
        }

        public Attempt withOnException(Runnable onFailure) {
            this.onException = Optional.of(onFailure);
            return this;
        }

        public boolean go() {
            return new Attempt(this).perform();
        }
    }
}

