package com.atlassian.beehive.db.spi;

import com.atlassian.beehive.core.ClusterLockStatus;
import com.atlassian.beehive.db.LockExpiryConfiguration;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.List;

/**
 * Provides access to the Cluster Lock DB table.
 * Clustered applications need to implement this DAO in order to be able to use the {@link com.atlassian.beehive.db.DatabaseClusterLockService}.
 */
public interface ClusterLockDao {

    /**
     * Returns lock details for all cluster locks being held by any cluster node.
     *
     * @return a list of cluster lock details for all locks held by any node, or an empty list if no such locks exist.
     */
    @Nonnull
    List<ClusterLockStatus> getAllHeldClusterLocks();

    /**
     * Returns the cluster lock details for the given name, or null if no such lock has ever been created.
     *
     * @param lockName the globally unique id of the lock
     * @return the cluster lock details for the given name, or null if no such lock has ever been created.
     */
    @Nullable
    ClusterLockStatus getClusterLockStatusByName(@Nonnull String lockName);

    /**
     * Attempts to acquire ownership of a named lock by current cluster node.
     * A lock should be treated as eligible to be acquired when one of the following is true:
     * <ul>
     *     <li>It has no node set as an owner</li>
     *     <li>It has the current node set as an owner</li>
     *     <li>It has an other node set as an owner, but its last update time is older than {@link LockExpiryConfiguration#getExpiryPeriodDurationSeconds()} from current db time</li>
     * </ul>
     *
     *  @param lockName the globally unique id of the lock
     *
     * @return true if we acquired the lock, false if the the locks already  has a valid owner.
     */
    boolean tryAcquireLock(@Nonnull String lockName);

    /**
     * Attempts to insert an "unlocked" DB row for the named lock.
     * (That is, the "locked by" column should be NULL).
     * <p>
     * You can assume that the lock name is not null.
     * This insert should only insert a single row for the given lock name (which will never be null).
     * This is expected to be achieved simply by putting a unique constraint on the LOCK_NAME column.
     * <p>
     * If the row already exists, then this method must catch the unique constraint violation and terminate normally.
     * All other SQL errors should be thrown as a RuntimeException of some kind.
     *
     * @param lockName   the lock identifier
     */
    void insertEmptyClusterLock(@Nonnull String lockName);

    /**
     * Unlock the named lock, if this node still holds the lock.
     *
     * If for some reason we no longer hold the lock then we should not unlock it, just report it by {@link IllegalMonitorStateException}.
     * Failure to perform the unlock because of enviromental problems should result with exception, to allow retries.
     *
     * @param lockName the lock identifier
     *
     */
    void unlock(@Nonnull String lockName);

    /**
     * Releases any locks held by the this node.
     *
     * @since 1.0
     */
    void releaseLocksHeldByNode();

    /**
     * Prolong the lock's lease, assuming it has current node set as the owner(but regardless of time passed since previous renewal).
     * If lock being no longer owned by this node is detected, it should be reported by {@link IllegalMonitorStateException}.
     *
     *  @param lockName The lock identifier
     *
     */
    void renewLease(@Nonnull String lockName);


}
