package com.atlassian.beehive.db;

import com.atlassian.beehive.core.stats.StatisticsKey;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.stream.LongStream;

import static com.atlassian.beehive.core.stats.StatisticsKey.AVERAGE_HOLD_TIME_MILLIS;
import static com.atlassian.beehive.core.stats.StatisticsKey.AVERAGE_WAIT_TIME_MILLIS;
import static com.atlassian.beehive.core.stats.StatisticsKey.ERROR;
import static com.atlassian.beehive.core.stats.StatisticsKey.FAIL_LOCAL;
import static com.atlassian.beehive.core.stats.StatisticsKey.FAIL_REMOTE;
import static com.atlassian.beehive.core.stats.StatisticsKey.FORCED_UNLOCK;
import static com.atlassian.beehive.core.stats.StatisticsKey.LAST_ACCESS;
import static com.atlassian.beehive.core.stats.StatisticsKey.LAST_ERROR;
import static com.atlassian.beehive.core.stats.StatisticsKey.LAST_FAIL_REMOTE;
import static com.atlassian.beehive.core.stats.StatisticsKey.LAST_LOCK;
import static com.atlassian.beehive.core.stats.StatisticsKey.LAST_RENEWAL;
import static com.atlassian.beehive.core.stats.StatisticsKey.LAST_STATE_ERROR;
import static com.atlassian.beehive.core.stats.StatisticsKey.LAST_UNLOCK;
import static com.atlassian.beehive.core.stats.StatisticsKey.STATE_ERROR;
import static com.atlassian.beehive.core.stats.StatisticsKey.SUCCESSFUL_LOCKS;
import static com.atlassian.beehive.core.stats.StatisticsKey.SUCCESSFUL_UNLOCKS;
import static com.atlassian.beehive.core.stats.StatisticsKey.WAIT_QUEUE_LENGTH;

/**
 * Statistics collection for a cluster lock.
 *
 * @since v0.2
 */
class StatisticsHolder
{

    public static final List<StatisticsKey> TIMESTAMPED_EVENTS_KEYS = ImmutableList.of(StatisticsKey.LAST_LOCK, StatisticsKey.LAST_FAIL_REMOTE,
            StatisticsKey.LAST_ERROR, StatisticsKey.LAST_STATE_ERROR, StatisticsKey.LAST_RENEWAL, StatisticsKey.LAST_UNLOCK);
    private final StatEvent unlock = new StatEvent();
    private final StatEvent lock = new StatEvent();
    private final StatEvent failByRemote = new StatEvent();
    private final StatEvent error = new StatEvent();
    private final StatEvent stateError = new StatEvent();

    private final AtomicLong failLocal = new AtomicLong();
    private final AtomicLong forcedUnlock = new AtomicLong();

    private final AtomicLong totalHoldTimeMillis = new AtomicLong();
    private final AtomicLong totalWaitTimeMillis = new AtomicLong();
    private final AtomicLong totalWaits = new AtomicLong();

    private final AtomicInteger waitQueueLength = new AtomicInteger();
    private final AtomicLong lastRenewalAt = new AtomicLong();

    private static void addAverageStat(final ImmutableMap.Builder<StatisticsKey, Long> stats, final StatisticsKey key,
            final Number total, final Number eventCount) {
        final long events = eventCount.longValue();
        if (events > 0L) {
            final long avg = total.longValue() / events;
            if (avg > 0L) {
                stats.put(key, avg);
            }
        }
    }

    /**
     * Print statistics in log friendly format.
     *
     * @param currentStatistics - statistics to print
     * @param currentTime       - point in time for printing how time passed since events in some statistics
     * @return log friendly formatted statistic message
     */
    public static String getStatisticsSummary(final Map<StatisticsKey, Long> currentStatistics, Long currentTime) {

        final String operationsPart = TIMESTAMPED_EVENTS_KEYS.stream()
                .filter(k -> currentStatistics.containsKey(k) && currentStatistics.get(k) > 0)
                .sorted(Comparator.comparingLong(currentStatistics::get).reversed())
                .map(k -> k.getLabel() + " " + (currentTime - currentStatistics.get(k)) + "ms ago")
                .collect(Collectors.joining(", "));

        final String statsPart = currentStatistics.keySet().stream()
                .filter(k -> !TIMESTAMPED_EVENTS_KEYS.contains(k))
                .filter(k -> currentStatistics.get(k) > 0)
                .map(k -> k.getLabel() + "=" + currentStatistics.get(k) + "")
                .collect(Collectors.joining(", "));
        StringBuffer buffer = new StringBuffer("Last events on this node at ");
        buffer.append(currentTime);
        buffer.append(": [");
        buffer.append(operationsPart);
        buffer.append("] Lock statistics: [");
        buffer.append(statsPart);
        buffer.append("]");
        return buffer.toString();
    }

    /**
     * Tally a {@link StatisticsKey#FAIL_LOCAL} event.
     */
    void tallyFailLocal() {
        failLocal.incrementAndGet();
    }

    /**
     * Tally a {@link StatisticsKey#FAIL_REMOTE} event.
     */
    void tallyFailRemote(Long timestamp) {
        failByRemote.onEvent(timestamp);
    }

    /**
     * Tally a {@link StatisticsKey#FORCED_UNLOCK} event.
     */
    void tallyForcedUnlock() {
        forcedUnlock.incrementAndGet();
    }

    /**
     * Tally a {@link StatisticsKey#ERROR} event.
     *
     * @param timestamp
     */
    void tallyError(final long timestamp) {
        error.onEvent(timestamp);
    }

    /**
     * Tally a {@link StatisticsKey#STATE_ERROR} event.
     *
     * @param timestamp
     */
    void tallyStateError(final long timestamp) {
        stateError.onEvent(timestamp);
    }

    /**
     * Retrieve the current values of the statistics for this cluster lock.
     *
     * @return the mapping of keys to values for all of this cluster lock's positive-valued statistics
     */
    Map<StatisticsKey, Long> getStatistics()
    {
        final ImmutableMap.Builder<StatisticsKey,Long> stats = ImmutableMap.builder();
        addStat(stats, FAIL_LOCAL, failLocal);
        addStat(stats, FAIL_REMOTE, failByRemote.getEventCount());
        addStat(stats, LAST_FAIL_REMOTE, failByRemote.getLastEventAt());
        addStat(stats, FORCED_UNLOCK, forcedUnlock);
        addStat(stats, ERROR, error.getEventCount());
        addStat(stats, STATE_ERROR, stateError.getEventCount());
        addStat(stats, LAST_ERROR, error.getLastEventAt());
        addStat(stats, LAST_STATE_ERROR, stateError.getLastEventAt());
        addStat(stats, LAST_LOCK, lock.getLastEventAt());
        addStat(stats, LAST_UNLOCK, unlock.getLastEventAt());
        addStat(stats, LAST_RENEWAL, lastRenewalAt.get());
        addStat(stats, LAST_ACCESS,
                LongStream.of(lock.getLastEventAt(),
                        lastRenewalAt.get(),
                        unlock.getLastEventAt(),
                        error.getLastEventAt(),
                        stateError.getLastEventAt(),
                        failByRemote.getLastEventAt()).
                        max().getAsLong());
        addStat(stats, SUCCESSFUL_LOCKS, lock.getEventCount());
        addStat(stats, SUCCESSFUL_UNLOCKS, unlock.getEventCount());
        addStat(stats, WAIT_QUEUE_LENGTH, waitQueueLength);
        addAverageStat(stats, AVERAGE_HOLD_TIME_MILLIS, totalHoldTimeMillis, unlock.getEventCount());
        addAverageStat(stats, AVERAGE_WAIT_TIME_MILLIS, totalWaitTimeMillis, totalWaits);
        return stats.build();
    }

    /**
     * Tally a {@link StatisticsKey#SUCCESSFUL_LOCKS} event and update {@link StatisticsKey#LAST_ACCESS}.
     *
     * @param currentTimeMillis the current time, expressed as a standard {@code System.currentTimeMillis()} timestamp.
     */
    void tallyLockedAt(final long currentTimeMillis) {
        lock.onEvent(currentTimeMillis);
    }

    /**
     * Adjusts the {@link StatisticsKey#WAIT_QUEUE_LENGTH} statistic to include this newly waiting thread.
     */
    void tallyWaitBegin()
    {
        waitQueueLength.incrementAndGet();
    }

    /**
     * Adjusts the {@link StatisticsKey#WAIT_QUEUE_LENGTH} and {@link StatisticsKey#AVERAGE_WAIT_TIME_MILLIS} to
     * reflect that this thread has finished waiting for the lock.
     * <p>
     * This should be called regardless of whether the wait for the lock was successful, timed out, or interrupted.
     * </p>
     */
    void tallyWaitEndAfter(final long durationInMillis)
    {
        waitQueueLength.decrementAndGet();
        if (durationInMillis > 0L)
        {
            totalWaitTimeMillis.addAndGet(durationInMillis);
        }

        // There is a tiny data race here, as the AVERAGE_WAIT_TIME_MILLIS is calculated from these two values and
        // they are not accessed as an atomic pair.  That does not seem important enough to warrant the complexity
        // of preventing it.
        totalWaits.incrementAndGet();
    }



    private static void addStat(final ImmutableMap.Builder<StatisticsKey,Long> stats, final StatisticsKey key,
            final Number valueHolder)
    {
        final long value = valueHolder.longValue();
        if (value > 0L)
        {
            stats.put(key, value);
        }
    }

    void tallyRenewed(final long timestamp) {
        lastRenewalAt.set(timestamp);
    }

    /**
     * Adjusts the {@link StatisticsKey#AVERAGE_HOLD_TIME_MILLIS} to include the time spent between the
     * values of {@code timestamp} provided to {@link #tallyLockedAt(long)} and the one provided
     * here.
     *
     * @param timestamp the current time, expressed as a standard {@code System.timestamp()} timestamp.
     */
    void tallyUnlockedAt(final long timestamp) {
        final long since = lock.getLastEventAt();
        unlock.onEvent(timestamp);
        if (timestamp > since) {
            totalHoldTimeMillis.addAndGet(timestamp - since);
        }
        // There is a tiny data race here, as the AVERAGE_HOLD_TIME_MILLIS is calculated from these two values and
        // they are not accessed as an atomic pair.  That does not seem important enough to warrant the complexity
        // of preventing it.
    }

    private static class StatEvent {

        private final AtomicLong lastEventAt = new AtomicLong();
        private final AtomicLong eventCount = new AtomicLong();

        public long getEventCount() {
            return eventCount.get();
        }

        public long getLastEventAt() {
            return lastEventAt.get();
        }

        public void onEvent(Long timestamp) {
            lastEventAt.set(timestamp);
            eventCount.incrementAndGet();
        }
    }
}
