package com.atlassian.beehive.db;

import com.atlassian.beehive.ClusterLock;
import com.atlassian.beehive.core.ClusterLockStatus;
import com.atlassian.beehive.core.LockRegistry;
import com.atlassian.beehive.core.ManagedClusterLockService;
import com.atlassian.beehive.db.spi.ClusterLockDao;

import javax.annotation.Nonnull;
import javax.annotation.PreDestroy;
import java.util.Collection;

/**
 * This is the database backed implementation of ClusterLockService that can be used by clustered applications.
 * <p>
 *     This implementation relies on the host application providing SPI implementations for Data Access objects.
 * </p>
 */
public class DatabaseClusterLockService implements ManagedClusterLockService
{
    private final ClusterLockDao clusterLockDao;
    private final LockRegistry<DatabaseClusterLock> registry = new LockRegistry<DatabaseClusterLock>()
    {
        @Override
        protected DatabaseClusterLock createLock(String lockName)
        {
            return new DatabaseClusterLock(lockName, clusterLockDao, databaseClusterLockLeaseRenewer);
        }
    };
    private final DatabaseClusterLockLeaseRenewer databaseClusterLockLeaseRenewer;


    public DatabaseClusterLockService(final ClusterLockDao clusterLockDao)
    {
        this.clusterLockDao = clusterLockDao;
        this.databaseClusterLockLeaseRenewer = new DatabaseClusterLockLeaseRenewer();
        clusterLockDao.releaseLocksHeldByNode();
    }


    @Override
    public Collection<DatabaseClusterLock> getAllKnownClusterLocks()
    {
        return registry.getAllKnownClusterLocks();
    }

    @Nonnull
    @Override
    public Collection<ClusterLockStatus> getStatusesOfAllHeldClusterLocks() {
        return clusterLockDao.getAllHeldClusterLocks();
    }

    @Override
    public ClusterLock getLockForName(@Nonnull String lockName)
    {
        return registry.getLockForName(lockName);
    }

    @PreDestroy
    public void shutdown() {
        databaseClusterLockLeaseRenewer.shutdown();
    }
}

