/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.beehive.db;

import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.beehive.db.DatabaseClusterLock;
import com.atlassian.beehive.db.LockExpiryConfiguration;
import com.atlassian.beehive.db.StatisticsHolder;
import com.atlassian.beehive.db.spi.ClusterLockDao;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class DatabaseClusterLockLeaseRenewer {
    private final ScheduledThreadPoolExecutor executor = new ScheduledThreadPoolExecutor(1, runnable -> {
        Thread t = new Thread(runnable, "cluster-lock-lease-renewal-thread");
        t.setDaemon(true);
        return t;
    });
    private final ConcurrentHashMap<String, ScheduledFuture> renewerMap;
    private final ClusterLockDao clusterLockDao;
    private static final Logger log = LoggerFactory.getLogger(DatabaseClusterLockLeaseRenewer.class);

    DatabaseClusterLockLeaseRenewer(ClusterLockDao clusterLockDao) {
        this.executor.setRemoveOnCancelPolicy(true);
        this.renewerMap = new ConcurrentHashMap();
        this.clusterLockDao = clusterLockDao;
    }

    void onLock(DatabaseClusterLock lock) {
        log.trace("onLock: " + lock.getName());
        RenewalTask task = this.createRenewalTask(lock);
        ScheduledFuture<?> future = this.executor.scheduleAtFixedRate(task, 5000L, LockExpiryConfiguration.getRenewalIntervalInSeconds() * 1000L, TimeUnit.MILLISECONDS);
        ScheduledFuture<?> previous = this.renewerMap.put(lock.getName(), future);
        if (previous != null) {
            log.warn("Detected previous, unfinished job when scheduling new renewer job for " + lock.getName() + ", cancelling old job...");
            previous.cancel(false);
        }
    }

    void onUnlock(DatabaseClusterLock lock) {
        log.trace("onUnlock: " + lock.getName());
        this.unschedule(lock.getName());
    }

    void shutdown() {
        this.executor.shutdownNow();
    }

    @VisibleForTesting
    RenewalTask createRenewalTask(DatabaseClusterLock lock) {
        return new RenewalTask(lock);
    }

    private void unschedule(String lockName) {
        ScheduledFuture future = this.renewerMap.remove(lockName);
        if (future != null) {
            future.cancel(false);
        }
    }

    class RenewalTask
    implements Runnable {
        private final DatabaseClusterLock lock;
        private final Long startedAt = System.currentTimeMillis();

        public RenewalTask(DatabaseClusterLock lock) {
            this.lock = lock;
        }

        @Override
        public void run() {
            log.trace("Renewing lease on lock: " + this.lock.getName());
            if (this.lock.isLockedLocally()) {
                try {
                    DatabaseClusterLockLeaseRenewer.this.clusterLockDao.renewLease(this.lock.getName());
                    this.lock.onRenewed();
                    log.trace("Successfully renewed lease on lock: {}, held for {} ", (Object)this.lock.getName(), (Object)(System.currentTimeMillis() - this.startedAt));
                }
                catch (IllegalMonitorStateException ex) {
                    String errorReport = StatisticsHolder.getStatisticsSummary(this.lock.getStatistics(), this.lock.nowInMillis());
                    log.error("Failed to renew lease on lock: " + this.lock.getName() + "  " + errorReport, (Throwable)ex);
                    this.lock.onStateError();
                    this.lock.interruptOwner();
                    this.unregisterSelf();
                }
                catch (Throwable t) {
                    String errorReport = StatisticsHolder.getStatisticsSummary(this.lock.getStatistics(), this.lock.nowInMillis());
                    log.error("Failed to renew lease on lock: , " + this.lock.getName() + " " + errorReport, t);
                    this.lock.onError();
                }
            } else {
                this.unregisterSelf();
            }
        }

        public void unregisterSelf() {
            DatabaseClusterLockLeaseRenewer.this.unschedule(this.getLockName());
        }

        private String getLockName() {
            return this.lock.getName();
        }
    }
}

