package com.atlassian.beehive.core.stats;

import java.util.Comparator;

import static com.atlassian.beehive.core.stats.StatisticsType.COUNTER;
import static com.atlassian.beehive.core.stats.StatisticsType.GAUGE;

/**
 * Represents some type of information that might be collected for a {@link com.atlassian.beehive.core.ManagedClusterLock}.
 * <p>
 * <strong>Warning</strong>: The caller should not make assumptions about which specific
 * statistics are available or what they mean.  All statistics are optional; some may be
 * omitted if they have an uninteresting value (like {@code 0}); and in many cases the
 * exact meaning has been left open for the implementation to define.  The statistics may
 * represent only local information as opposed to reporting the global state of the cluster.
 * </p>
 * <p>
 * Developers are encouraged to disclose any design decisions they have made in the
 * JavaDocs for the {@link com.atlassian.beehive.core.ManagedClusterLock#getStatistics()}
 * implementation.
 * </p>
 *
 * @since v0.2
 */
public enum StatisticsKey
{
    /**
     * The average length of time, in milliseconds, that the lock is held after it has
     * been acquired.
     */
    AVERAGE_HOLD_TIME_MILLIS("averageHoldTimeMillis", GAUGE),

    /**
     * The average length of time, in milliseconds, that threads have had to block waiting
     * to acquire the lock, whether they were ultimately successful or not.
     */
    AVERAGE_WAIT_TIME_MILLIS("averageWaitTimeMillis", GAUGE),

    /**
     * The total number of times that a request to acquire the lock failed because it was
     * already held by another thread on the same node.
     */
    FAIL_LOCAL("failLocal", COUNTER),

    /**
     * The total number of times that a request to acquire the lock failed because it was
     * already held by another node in the cluster.
     */
    FAIL_REMOTE("failRemote", COUNTER),

    /**
     * A timestamp, expressed in millis since the epoch, when a request to acquire the lock failed because it was
     * already held by another node in the cluster.
     */

    LAST_FAIL_REMOTE("lastFailRemote", COUNTER),

    /**
     * A timestamp, expressed in millis since the epoch, when a last request to acquire/release/renew the lock failed because of a lock state state exception
     */
    LAST_STATE_ERROR("lastStateError", COUNTER),

    /**
     * The total number of times that a request to acquire/release/renew the lock failed because of a lock state related exception
     */
    STATE_ERROR("failByStateError", COUNTER),

    /**
     * A timestamp, expressed in millis since the epoch, when a last request to acquire/release/renew the lock failed because of a non-state-related exception
     */
    LAST_ERROR("lastError", COUNTER),

    /**
     * The total number of times that a request to acquire/release/renew the lock failed because of a non-state-related exception
     */
    ERROR("failByError", COUNTER),

    /**
     * A timestamp, expressed in millis since the epoch, when this lock was last accessed.
     */
    LAST_ACCESS("lastAccess", GAUGE),

    /**
     * A timestamp, expressed in millis since the epoch, when this lock was last successfully locked.
     */
    LAST_LOCK("lastLock", GAUGE),

    /**
     * A timestamp, expressed in millis since the epoch, when this lock was last unlocked.
     */
    LAST_UNLOCK("lastUnlock", GAUGE),

    /**
     * A timestamp, expressed in millis since the epoch, when this lock's state was last renewed.
     */
    LAST_RENEWAL("lastRenewal", GAUGE),

    /**
     * The total number of times that a lock has had to be "recovered" by forcibly unlocking it.
     */
    FORCED_UNLOCK("forcedUnlock", COUNTER),

    /**
     * The total number of times that the lock has been successfully acquired.
     */
    SUCCESSFUL_LOCKS("lockSuccess", COUNTER),

    /**
     * The total number of times that the lock has been released(regardless of exceptions that happended in the process).
     */
    SUCCESSFUL_UNLOCKS("unlockSuccess", COUNTER),

    /**
     * The number of threads that are currently waiting on this lock.
     */
    WAIT_QUEUE_LENGTH("waitQueueLength", GAUGE);


    /** Sorts statistics keys by {@link #getLabel() label}. */
    public static Comparator<StatisticsKey> SORT_BY_LABEL = new Comparator<StatisticsKey>()
    {
        @Override
        public int compare(StatisticsKey key1, StatisticsKey key2)
        {
            return key1.getLabel().compareTo(key2.getLabel());
        }
    };

    private final String label;
    private final StatisticsType type;

    private StatisticsKey(final String label, final StatisticsType type)
    {
        this.label = label;
        this.type = type;
    }

    /**
     * Returns the label for this statistics key.
     * This is intended to be the same as the {@link #name()}, except written in {@code camelCase}.
     * @return the label for this statistics key.
     */
    public String getLabel()
    {
        return label;
    }

    /**
     * Returns the type of statistic that this is.
     * @return the type of statistic that this is.
     */
    public StatisticsType getType()
    {
        return type;
    }
}
