package com.atlassian.beehive.core;

import java.util.Collection;

import com.atlassian.beehive.ClusterLockService;

import javax.annotation.Nonnull;

/**
 * Decorates a cluster lock service with management concerns, such as instrumentation.
 *
 * @since v0.2
 */
public interface ManagedClusterLockService extends ClusterLockService
{
    /**
     * Returns all the cluster locks that are currently known to this cluster lock service.
     * <p>
     * The specifics are mostly left up to the implementation, with the following caveats:
     * </p>
     * <ul>
     * <li>It <strong>MUST</strong> include all locks that are currently locked by this node.</li>
     * <li>It <strong>SHOULD</strong> include all locks that are still strongly referenced outside
     *          of the cluster lock service, if possible.  Implementations that are unable to comply
     *          with this requirement <strong>MUST</strong> clearly document that limitation.</li>
     * <li>It <strong>MAY</strong> include locks that were previously accessed but are no longer in use,
     *          even those that are no longer strongly referenced outside of the cluster lock service.</li>
     * <li>It <strong>MAY</strong> include locks that have not been accessed by this node but are still
     *          known to the service through the activity of other nodes within the cluster.</li>
     * </ul>
     * <p>
     * Consumers <strong>MUST NOT</strong> rely on this method for the correctness of their locking behaviour.
     * This information is intended for instrumentation and other similar management concerns, only.
     * </p>
     *
     * @return all cluster locks that are currently known to this service, in no particular order
     */
    Collection<? extends ManagedClusterLock> getAllKnownClusterLocks();

    /**
     * Returns lock status of every cluster lock being held.
     * <p>
     * This is intended to provide additional information about misbehaving locks and thus this method
     *          should return the most up-to-date and non-cached data.
     * </p>
     * <p>
     * Consumers must be aware that not all implementations can return a valid update time of the locks.
     * </p>
     *
     * @return a collection of cluster lock statuses, in no particular order
     */
    @Nonnull
    Collection<ClusterLockStatus> getStatusesOfAllHeldClusterLocks();
}
