package com.atlassian.beehive.core;

import java.util.Collection;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import com.atlassian.beehive.ClusterLock;
import com.atlassian.beehive.ClusterLockService;

import com.google.common.collect.ImmutableList;

/**
 * Provides a lock registry for {@code ManagedClusterLockService} implementations.
 * <p>
 * The lock registry maintains a unique mapping of lock names to corresponding {@link ClusterLock} objects on
 * behalf of a {@link ManagedClusterLockService}.  Lock services that use this registry to manage this mapping
 * should implement {@link ManagedClusterLockService#getAllKnownClusterLocks()},
 * {@link ManagedClusterLockService#getAllKnownClusterLocks}, and {@link ClusterLockService#getLockForName(String)} by
 * delegating them to the lock registry.
 * </p>
 *
 * @since v0.2
 * @param <T> the type of {@code ManagedClusterLock} that this registry tracks
 * @see #createLock(String)
 */
public abstract class LockRegistry<T extends ManagedClusterLock>
{
    private final ConcurrentMap<String,T> lockMap = new ConcurrentHashMap<>();

    /**
     * A delegate implementation for {@link ManagedClusterLockService#getAllKnownClusterLocks()} that will return
     * all registered {@code ManagedClusterLock} instances that have previously been returned by calls to
     * {@link #getLockForName(String)}.
     *
     * @return an immutable collection of all the known cluster locks, in no particular order
     */
    public Collection<T> getAllKnownClusterLocks()
    {
        return ImmutableList.copyOf(lockMap.values());
    }

    /**
     * A delegate implementation for {@link ManagedClusterLockService#getStatusesOfAllHeldClusterLocks()} that will
     * return {@link com.atlassian.beehive.core.ClusterLockStatus} for every lock being held.
     *
     * @return an immutable collection of cluster lock statuses, in no particular order
     */
    public Collection<ClusterLockStatus> getStatusesOfAllHeldClusterLocks() {
        return ImmutableList.copyOf(lockMap.values().stream()
                .filter(ManagedClusterLock::isLocked)
                .map(ManagedClusterLock::getClusterLockStatus)
                .collect(Collectors.toList()));
    }

    /**
     * A delegate implementation for {@link ManagedClusterLockService#getLockForName(String)} that will ensure that
     * the same lock instance is returned on every call.
     *
     * @param lockName the name of the lock
     * @return a new lock if this {@code lockName} has not previously been requested; otherwise, the same lock instance
     *          that was previously returned for it
     */
    public ClusterLock getLockForName(@Nonnull final String lockName)
    {
        // normally it should already be in the map
        T lock = lockMap.get(lockName);
        if (lock != null)
        {
            return lock;
        }

        // We need to create a lock.  However, if another thread wins the race to claim the map entry, use theirs.
        lock = createLock(lockName);
        final T sneak = lockMap.putIfAbsent(lockName, lock);
        return (sneak != null) ? sneak : lock;
    }

    /**
     * Creates a new cluster lock with the given name.
     * <p>
     * <strong>WARNING</strong>: This method may be called by multiple threads at once.  The implementation
     * of {@link #getLockForName(String)} will ensure that only one of the created lock instances is ever
     * returned, but this factory method must create lock objects for which creating an extra lock of the same name
     * causes no problems so long as the extra lock is never actually used.  If this in not possible, then the
     * cluster lock service should implement {@link ManagedClusterLockService#getLockForName(String)} and
     * {@link ManagedClusterLockService#getAllKnownClusterLocks()} directly instead of using this lock registry.
     * </p>
     *
     * @param lockName the name of the lock to create
     * @return the newly create lock, which may or may not actually end up being used
     */
    protected abstract T createLock(String lockName);
}

