/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.beehive;

import com.atlassian.beehive.ClusterLock;
import com.atlassian.beehive.ClusterNodeHeartbeatService;
import com.atlassian.beehive.spi.ClusterLockDao;
import com.atlassian.beehive.spi.ClusterLockStatus;
import java.lang.ref.WeakReference;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Condition;
import javax.annotation.Nonnull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class DatabaseClusterLock
implements ClusterLock {
    private static final long RECHECK_INTERVAL_MILLIS = 10000L;
    private static final int INITIAL_SLEEP_MILLIS = 100;
    private static final int MAX_SLEEP_MILLIS = 10000;
    private static final Logger log = LoggerFactory.getLogger(DatabaseClusterLock.class);
    private final String lockName;
    private final ClusterLockDao clusterLockDao;
    private final ClusterNodeHeartbeatService clusterNodeHeartbeatService;
    private final String nodeId;
    private final AtomicReference<Owner> ownerRef = new AtomicReference();
    private final AtomicInteger depth = new AtomicInteger();
    private final AtomicLong lastCheck = new AtomicLong();

    public DatabaseClusterLock(String lockName, ClusterLockDao clusterLockDao, ClusterNodeHeartbeatService clusterNodeHeartbeatService) {
        this.lockName = lockName;
        this.clusterLockDao = clusterLockDao;
        this.clusterNodeHeartbeatService = clusterNodeHeartbeatService;
        this.nodeId = clusterNodeHeartbeatService.getNodeId();
    }

    public void lock() {
        boolean wasInterrupted = false;
        while (true) {
            try {
                this.lockInterruptibly();
                if (wasInterrupted) {
                    Thread.currentThread().interrupt();
                }
                return;
            }
            catch (InterruptedException e) {
                wasInterrupted = true;
                continue;
            }
            break;
        }
    }

    public void lockInterruptibly() throws InterruptedException {
        int sleepTimeMillis = 100;
        while (!this.tryLock()) {
            Thread.sleep(sleepTimeMillis);
            sleepTimeMillis = Math.min(sleepTimeMillis * 2, 10000);
        }
    }

    public boolean tryLock() {
        Thread me = Thread.currentThread();
        log.debug("Attempt to get cluster lock '{}' by {}.", (Object)this.lockName, (Object)me);
        Owner owner = this.ownerRef.get();
        if (owner == null || !this.isLocalPermitValid(owner)) {
            return this.tryLockUsingDatabase();
        }
        if (owner.getThread() == me) {
            log.debug("Cluster lock '{}' reentered by '{}'", (Object)this.lockName, (Object)me);
            if (this.depth.incrementAndGet() < 0) {
                this.depth.decrementAndGet();
                throw new IllegalMonitorStateException("Maximum lock count exceeded");
            }
            return true;
        }
        Thread ownerThread = owner.getThread();
        if (ownerThread != null && ownerThread.isAlive()) {
            log.debug("Cluster lock '{}' currently held by another local thread '{}'.", (Object)this.lockName, (Object)ownerThread.getName());
            return false;
        }
        if (this.ownerRef.compareAndSet(owner, new Owner(me))) {
            log.error("Cluster lock '{}' was not unlocked by '{}' before it terminated, so '{}' has stolen it", new Object[]{this.lockName, owner, me});
            this.depth.set(1);
            this.lastCheck.set(System.currentTimeMillis());
            return true;
        }
        log.debug("Cluster lock '{}' was stolen by another thread, so '{}' lost this race", (Object)this.lockName, (Object)me);
        return false;
    }

    private boolean tryLockUsingDatabase() {
        ClusterLockStatus clusterLockStatus = this.getClusterLockStatus();
        if (clusterLockStatus.getLockedByNode() != null && !clusterLockStatus.getLockedByNode().equals(this.nodeId)) {
            log.debug("Cluster lock '{}' currently held by node '{}'.", (Object)this.lockName, (Object)clusterLockStatus.getLockedByNode());
            long age = System.currentTimeMillis() - clusterLockStatus.getUpdateTime();
            if (age > 300000L) {
                clusterLockStatus = this.unlockIfDead(clusterLockStatus);
            }
        }
        Thread me = Thread.currentThread();
        if (clusterLockStatus.getLockedByNode() == null && this.clusterLockDao.tryUpdateAcquireLock(this.lockName, this.nodeId, System.currentTimeMillis())) {
            this.ownerRef.set(new Owner(me));
            this.depth.set(1);
            this.lastCheck.set(System.currentTimeMillis());
            log.debug("Cluster lock '{}' was acquired by {}.", (Object)this.lockName, (Object)me);
            return true;
        }
        log.debug("Acquisition of cluster lock '{}' by {} failed.", (Object)this.lockName, (Object)me);
        return false;
    }

    public boolean tryLock(long waitTime, @Nonnull TimeUnit unit) throws InterruptedException {
        long timeout = System.currentTimeMillis() + unit.toMillis(waitTime);
        if (Thread.interrupted()) {
            throw new InterruptedException();
        }
        long sleepTimeMillis = 100L;
        while (!this.tryLock()) {
            long remainingWaitTime = timeout - System.currentTimeMillis();
            if (remainingWaitTime <= 0L) {
                return false;
            }
            sleepTimeMillis = Math.min(sleepTimeMillis, remainingWaitTime);
            Thread.sleep(sleepTimeMillis);
            sleepTimeMillis = Math.min(sleepTimeMillis * 2L, 10000L);
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void unlock() {
        Owner owner = this.ownerRef.get();
        Thread me = Thread.currentThread();
        if (owner == null || owner.getThread() != me || !this.isLocalPermitValid(owner)) {
            throw new IllegalMonitorStateException("Cluster lock '" + this.lockName + "' cannot be unlocked because it is not owned by this thread: " + me);
        }
        if (this.depth.decrementAndGet() > 0) {
            log.debug("Cluster lock '{}' re-entrance count decremented by '{}'", (Object)this.lockName, (Object)me);
            return;
        }
        log.debug("Cluster lock '{}' unlocked by '{}'", (Object)this.lockName, (Object)me);
        try {
            this.clusterLockDao.unlock(this.lockName, this.nodeId, System.currentTimeMillis());
        }
        finally {
            this.ownerRef.compareAndSet(owner, null);
        }
    }

    public boolean isHeldByCurrentThread() {
        Owner owner = this.ownerRef.get();
        return owner != null && owner.getThread() == Thread.currentThread() && this.isLocalPermitValid(owner);
    }

    private boolean isLocalPermitValid(@Nonnull Owner owner) {
        long now = System.currentTimeMillis();
        if (now - this.lastCheck.get() < 10000L) {
            return true;
        }
        ClusterLockStatus clusterLockStatus = this.getClusterLockStatus();
        if (this.nodeId.equals(clusterLockStatus.getLockedByNode())) {
            this.lastCheck.set(now);
            return true;
        }
        if (this.ownerRef.compareAndSet(owner, null)) {
            log.error("Cluster lock '{}' was expected to already be held by this node, but its current owner is '{}'.", (Object)this.lockName, (Object)clusterLockStatus.getLockedByNode());
        }
        return false;
    }

    @Nonnull
    public Condition newCondition() {
        throw new UnsupportedOperationException("newCondition() not supported in ClusterLock");
    }

    private ClusterLockStatus getClusterLockStatus() {
        ClusterLockStatus lock = this.clusterLockDao.getClusterLockStatusByName(this.lockName);
        if (lock != null) {
            return lock;
        }
        long updateTime = System.currentTimeMillis();
        this.clusterLockDao.insertEmptyClusterLock(this.lockName, updateTime);
        return new ClusterLockStatus(this.lockName, null, updateTime);
    }

    private ClusterLockStatus unlockIfDead(ClusterLockStatus lock) {
        if (this.clusterNodeHeartbeatService.isNodeLive(lock.getLockedByNode())) {
            return lock;
        }
        log.warn("Releasing lock '" + lock.getLockName() + "' from node '" + lock.getLockedByNode() + "' because the node has stopped heart-beating.");
        long updateTime = System.currentTimeMillis();
        this.clusterLockDao.unlock(lock.getLockName(), lock.getLockedByNode(), updateTime);
        return new ClusterLockStatus(this.lockName, null, updateTime);
    }

    static class Owner {
        private final WeakReference<Thread> thd;
        private final String name;

        Owner(Thread me) {
            this.thd = new WeakReference<Thread>(me);
            this.name = me.getName();
        }

        Thread getThread() {
            return (Thread)this.thd.get();
        }

        public String toString() {
            return this.name;
        }
    }
}

