define('branchinator/views/layout/header/branch-selection-view', [
    'backbone',
    'branchinator/utils/branchinator-utils',
    'jquery',
    'branchinator/utils/radio-utils'
], function(
    Backbone,
    bnUtils,
    $,
    radioUtils
) {
    'use strict';

    /**
     * Branch selector mode when no repository is selected. For a given search string, we'll search for available
     * branches across various linked repositories for all build plans in Bamboo.
     *
     * @param $branchSelector UI element of the branch selector
     *
     * @constructor creates new instance of this selector mode
     */
    var WithoutRepoBranchSelectionMode = function($branchSelector) {
        $branchSelector.data('select2').opts.minimumInputLength = 2;
    };

    WithoutRepoBranchSelectionMode.prototype = {
        url: function() {
            return bnUtils.getRestUrl('branchesWithPlans');
        },
        results: function (data) {
            // first record is not selectable and is used for displaying tooltip
            data.unshift({ disabled: true });
            return { results: data };
        },
        formatResult: function (result) {
            if (result.disabled) {
                return AJS.I18n.getText('branchinator.branch.use.repository.selector');
            } else {
                return bamboo.plugins.branchinator.renderBranchSelectorOption({
                    repoName: result.repoName,
                    branchName: result.branchName
                });
            }
        },
        fetchErrorMessage: function() {
            return AJS.I18n.getText('branchinator.branch.select.withoutRepo.fetchError');
        }
    };

    /**
     * Branch selector mode when a repository is selected. For a given search string, we'll only search for branches for
     * currently selected repository.
     *
     * @param $branchSelector UI element of the branch selector
     * @param branchSelectionModel model which stores currently selected repository and branch
     *
     * @constructor creates new instance of this selector mode
     */
    var WithRepoBranchSelectionMode = function($branchSelector, branchSelectionModel) {
        this.branchSelectionModel = branchSelectionModel;
        $branchSelector.data('select2').opts.minimumInputLength = 0;
    };

    WithRepoBranchSelectionMode.prototype = {
        getRepoId: function() {
            return this.branchSelectionModel.get('repo').id;
        },
        url: function() {
            return bnUtils.getRestUrl('branches?repoId={0}', this.getRepoId());
        },
        results: function(data) {
            return { results: data };
        },
        formatResult: function(result) {
            return bamboo.plugins.branchinator.renderBranchSelectorOption({ branchName: result.branchName });
        },
        fetchErrorMessage: function() {
            return AJS.I18n.getText('branchinator.branch.select.withRepo.fetchError');
        }
    };

    /**
     * View for displaying and managing repository and branch selectors.
     *
     * Parameters:
     * - model - branch selection model
     */
    return Marionette.ItemView.extend({
        tagName: 'span',

        ui: {
            repoSelector: '#repoSelector',
            branchSelector: '#branchSelector',
            copyLink: '#copyBranchNameLink'
        },

        modelEvents: {
            'repo:selected': 'onRepoSelected',
            'repo:cleared': 'onRepoCleared',
            'branch:selected': 'onBranchSelected',
            'branch:cleared': 'onBranchCleared'
        },

        events: {
            'change @ui.repoSelector': 'onRepoSelectorChange',
            'change @ui.branchSelector': 'onBranchSelectorChange'
        },

        /**
         * Branch selection operates in various modes, depending on whether repository is selected or not. This field
         * holds reference to currently active mode.
         */
        branchSelectionMode: void 0,

        template: function() {
            return bamboo.plugins.branchinator.renderBranchSelectionView();
        },

        onShow: function() {
            this.ui.repoSelector.auiSelect2({
                placeholder: AJS.I18n.getText('branchinator.repository.select'),
                minimumInputLength: 0,
                dropdownCssClass: 'repoSelectorDropdown',
                ajax: {
                    url: bnUtils.getRestUrl('repos'),
                    dataType: 'json',
                    data: function (term, page) {
                        return { searchTerm: term };
                    },
                    results: function (data, page) {
                        return { results: data };
                    },
                    cache: true,
                    params: {
                        error: this.onRepoFetchError.bind(this)
                    }
                },
                formatResult: function (repo) {
                    return repo.name;
                },
                formatSelection: function (repo) {
                    return repo.name;
                }
            });

            this.ui.branchSelector.auiSelect2({
                placeholder: AJS.I18n.getText('branchinator.branch.select'),
                minimumInputLength: 2,
                dropdownCssClass: 'branchSelectorDropdown',
                ajax: {
                    url: function() {
                        return this.branchSelectionMode.url();
                    }.bind(this),
                    dataType: 'json',
                    data: function(term) {
                        return { searchTerm: term };
                    },
                    results: function(data, page) {
                        return this.branchSelectionMode.results(data, page);
                    }.bind(this),
                    cache: false,
                    params: {
                        error: this.onBranchFetchError.bind(this)
                    }
                },
                id: function(branch) {
                    return branch.branchName;
                },
                formatResult: function(branch) {
                    return this.branchSelectionMode.formatResult(branch);
                }.bind(this),
                formatSelection: function(branch) {
                    return bamboo.plugins.branchinator.renderBranchSelection({ branch: branch.branchName });
                },
                formatInputTooShort: function(term, minLength) {
                    return AJS.I18n.getText('branchinator.branch.use.repository.inputTooShort');
                }
            });

            this.branchSelectionMode = new WithoutRepoBranchSelectionMode(this.ui.branchSelector);
        },

        onRepoSelectorChange: function(e) {
            var repository = this.ui.repoSelector.auiSelect2('data');
            this.model.selectRepo(repository.id, repository.name);
        },

        onBranchSelectorChange: function(e) {
            var branch = this.ui.branchSelector.auiSelect2('data');
            this.model.selectBranch(branch.repoId, branch.repoName, branch.branchName);
        },

        onRepoCleared: function() {
            this.ui.repoSelector.val('');
            this.ui.repoSelector.auiSelect2('data', '');
            this.branchSelectionMode = new WithoutRepoBranchSelectionMode(this.ui.branchSelector);
            this.ui.copyLink.empty();
        },

        onRepoSelected: function(selection) {
            this.branchSelectionMode = new WithRepoBranchSelectionMode(this.ui.branchSelector, this.model);
            this.ui.repoSelector.val(selection.id);
            this.ui.repoSelector.auiSelect2('data', {
                id: selection.id,
                name: selection.name
            });
            this.ui.copyLink.empty();
        },

        onBranchCleared: function() {
            this.ui.branchSelector.val('');
            this.ui.branchSelector.auiSelect2('data', '');
            this.ui.copyLink.empty();
        },

        onBranchSelected: function(selection) {
            this.onRepoSelected(selection.repo);

            var selectedBranch = selection.branch;
            var branchName = selectedBranch.branchName;
            this.ui.branchSelector.val(selectedBranch.id);
            this.ui.branchSelector.auiSelect2('data', {
                id: selectedBranch.id,
                branchName: branchName
            });
            this.ui.copyLink.html(bamboo.plugins.branchinator.renderCopyBranchLink({ branchName: branchName }));
        },

        onRepoFetchError: function(response) {
            if (response.status !== 0 || response.statusText !== 'abort') {
                var error = bnUtils.getErrorFromResponse(response, AJS.I18n.getText('branchinator.repository.select.fetchError'));
                radioUtils.trigger('message:error', error);
            }
        },

        onBranchFetchError: function(response) {
            if (response.status !== 0 || response.statusText !== 'abort') {
                var error = bnUtils.getErrorFromResponse(response, this.branchSelectionMode.fetchErrorMessage());
                radioUtils.trigger('message:error', error);
            }
        }
    });
});
