define('branchinator/utils/branchinator-utils', [], function() {
    'use strict';

    var applicationRootUrl = AJS.contextPath() + '/branchStatus';
    var restRootUrl = AJS.contextPath() + '/rest/branchStatus/latest/';

    // name of event fired when a CSS3 animation ends, depends on browser type
    var animationCompletedEventName = (function() {
        var fakeElement = document.createElement("fakeElement");

        // a mapping of 'supported CSS style' => 'fired event name'
        var animationEndEvents = {
            'animation': 'animationend',
            'OAnimation': 'oAnimationEnd',
            'MozAnimation': 'animationend',
            'WebkitAnimation': 'webkitAnimationEnd'
        };

        for (var styleName in animationEndEvents) {
            if (fakeElement.style[styleName] !== undefined) {
                return animationEndEvents[styleName];
            }
        }
    })();

    return Object.freeze({

        /**
         * Returns root URL to branch status page.
         */
        getApplicationRootUrl: function() {
            return applicationRootUrl;
        },

        /**
         * Creates a URL for branchinator REST resource. The result will include context path.
         *
         * First argument should be the relative path to resource location. Additional arguments will be used to format
         * the URL using AJS.format method.
         *
         * Example usages:
         * - bnUtils.getRestUrl('commits');
         * - bnUtils.getRestUrl('user/properties/{0}', someProperty);
         */
        getRestUrl: function() {
            return restRootUrl + AJS.format.apply(null, arguments);
        },

        /**
         * Returns URL path relative to branch status root page, pointing directly to the given repository.
         */
        getRepoUrlPath: function(repoId, repoName) {
            return (repoId && repoName) ? repoId + '/' + this.slashSafeUrlEncode(repoName) : '';
        },

        /**
         * Returns URL path relative to branch status root page, pointing directly to a given branch of a given
         * repository.
         */
        getRepoAndBranchUrlPath: function(repoId, repoName, branchName) {
            return this.getRepoUrlPath(repoId, repoName) + (branchName ? '/' + this.slashSafeUrlEncode(branchName) : '');
        },

        /**
         * This method will encode URL component while double-encoding slashes and backslashes.
         *
         * This is sometimes necessary because Tomcat and other Apache servlet containers don't accept encoded slashes
         * for security reasons.
         */
        slashSafeUrlEncode: function(urlComponent) {
            if (urlComponent) {
                return encodeURIComponent(urlComponent).replace(/%2[fF]/g, "%252F").replace(/%5C/, "%255C");
            }
        },

        /**
         * This method will decode URL component while double-decoding slashes and backslashes.
         *
         * This is sometimes necessary because Tomcat and other Apache servlet containers don't accept encoded slashes
         * for security reasons.
         */
        slashSafeUrlDecode: function(urlComponent) {
            if (urlComponent) {
                return urlComponent.replace(/%2[fF]/g, "/").replace(/%5C/, "\\");
            }
        },

        /**
         * Returns error message from XHQR response, if the response is in a known format. Returns general error message
         * otherwise.
         *
         * @param response jqXHR object
         * @param [defaultMessage] default message to be displayed if error message is not obtainable from the response
         */
        getErrorFromResponse: function(response, defaultMessage) {
            var responseJSON = response.responseJSON;
            var errorMessage = responseJSON && responseJSON.errors && responseJSON.errors[0];
            return errorMessage || defaultMessage || AJS.I18n.getText('branchinator.application.error');
        },

        /**
         * Returns name of a jQuery event which is fired when an element has finished it's animation. Only applies to
         * CSS3 animations (keyframes transformation).
         */
        getAnimationCompletedEventName: function() {
            return animationCompletedEventName;
        }
    });
});
