define('branchinator/routers/router', [
    'backbone',
    'branchinator/utils/branchinator-utils',
    'branchinator/utils/radio-utils',
    'util/uri-utils'
], function(
    Backbone,
    bnUtils,
    radioUtils,
    uriUtils
) {
    'use strict';

    var knownNavigationSources = WRM.data.claim('com.atlassian.bamboo.plugins.branchinator:branchinator-resources.known-navigation-sources');

    /**
     * Branchinator router for handling browser history and URL manipulation.
     *
     * Parameters:
     * - model - branch selection model
     *
     * TODO: after upgrading to Backbone 1.1.1, query params may be retrieved by matching routes, no need for uri-utils
     */
    return Backbone.Router.extend({

        routes: {
            "(/)": "selectNone",
            ":repoId/:repoName(/)" : "selectRepo",
            ":repoId/:repoName/:branchName(/)" : "selectRepoAndBranch"
        },

        initialize: function(options) {
            this.model = options.model;
            this.triggerAnalytics();
        },

        triggerAnalytics: function() {
            var queryObj = uriUtils.queryToObject();
            var source = queryObj.source && this.isSourceKnown(queryObj.source) ? queryObj.source : 'direct';
            AJS.trigger('analyticsEvent', { name: 'branch.view.page.opened', data: { source: source } });
        },

        isSourceKnown: function(navigationSource) {
            return knownNavigationSources.indexOf(navigationSource) > -1;
        },

        removeQueryParams: function() {
            if (window.location.search) {
                var fragment = Backbone.history.getFragment();
                this.navigate(fragment + '/clear', { replace: true });
                this.navigate(fragment, { replace: true });
            }
        },

        selectNone: function() {
            this.removeQueryParams();
            this.model.clearRepo();
        },

        selectRepo: function(repoId, repoName) {
            this.removeQueryParams();
            this.model.selectRepo(repoId, bnUtils.slashSafeUrlDecode(repoName));
            this.model.clearBranch();
        },

        selectRepoAndBranch: function(repoId, repoName, branchName) {
            this.removeQueryParams();
            this.model.selectBranch(repoId, bnUtils.slashSafeUrlDecode(repoName), bnUtils.slashSafeUrlDecode(branchName));
        }
    });
});
