define('branchinator/collections/pull-requests-collection', [
    'backbone',
    'branchinator/utils/branchinator-utils',
    'branchinator/models/pull-request-model',
    'underscore'
], function(
    Backbone,
    bnUtils,
    PullRequestModel,
    _
) {
    'use strict';

    /**
     * Collection of pull requests from a given VCS branch.
     */
    return Backbone.Collection.extend({
        model: PullRequestModel,

        initialize: function(models, options) {
            this.repoId = options.repoId;
            this.branchName = options.branchName;

            this.resetMergeabilityStates();

            if (models) {
                models.forEach(this.normalizePullRequestObject, this);
            }

            this.fetchMergeabilityStates();
            this.on('sync', this.fetchMergeabilityStates);
            this.on('remove', function(model) { this.removeMergeabilityState(model.getId()); });
            this.on('reset', this.resetMergeabilityStates);
        },

        parse: function(data) {
            var paginatedResult = data.pullRequests || {};
            var pullRequests = paginatedResult.values || [];
            pullRequests.forEach(this.normalizePullRequestObject, this);
            return pullRequests;
        },

        /**
         * Normalizes the object to be stored in this collection. The param should be a simple JavaScript object, not
         * an instance of this collection's model.
         *
         * @param pullRequest {Object}
         */
        normalizePullRequestObject: function(pullRequest) {
            // set Bamboo repository ID for each model
            pullRequest[PullRequestModel.prototype.bambooRepoIdAttribute] = this.repoId;

            // set previously known mergeability state for each model
            var pullRequestId = pullRequest[PullRequestModel.prototype.idAttribute];
            if (pullRequestId !== void 0) {
                _.extend(pullRequest, this.getMergeabilityState(pullRequestId));
            }
        },

        /**
         * Obtain mergeability state of all pull requests in this collection from Bamboo server. Fetched data is saved
         * in a local cache variable for future usage.
         */
        fetchMergeabilityStates: function() {
            this.models.forEach(function(model) {
                model.fetchMergeabilityState().done(function() {
                    this._mergeabilityState[model.getId()] = model.getMergeabilityState();
                }.bind(this));
            }, this);
        },

        /**
         * Retrieves mergeability state of a single pull request.
         * @param pullRequestId {number}
         */
        getMergeabilityState: function(pullRequestId) {
            return this._mergeabilityState[pullRequestId] || {};
        },

        /**
         * Removes mergeability state for the selected pull request.
         * @param pullRequestId {number}
         */
        removeMergeabilityState: function(pullRequestId) {
            delete this._mergeabilityState[pullRequestId];
        },

        /**
         * Reset mergeability states cache.
         */
        resetMergeabilityStates: function() {
            this._mergeabilityState = {};
        }

    }, {

        /**
         * Creates an instance of pull requests collection for the given repository and branch.
         *
         * @param repoId {number} id of repository in Bamboo
         * @param branchName {string} name of VCS branch
         */
        forRepositoryBranch: function(repoId, branchName) {
            return new this([], {
                url: bnUtils.getRestUrl('pullRequest?repoId={0}&branchName={1}',
                    repoId, encodeURIComponent(branchName)),
                repoId: repoId,
                branchName: branchName
            });
        }
    });
});
