define('branchinator/collections/builds-collection', [
    'backbone',
    'branchinator/utils/branchinator-utils',
    'branchinator/models/build-model',
    'underscore'
], function(
    Backbone,
    bnUtils,
    BuildModel,
    _
) {
    'use strict';

    /**
     * Collection of build plans for the selected repository and branch. Each element contains information about plan
     * itself and it's latest result.
     */
    return Backbone.Collection.extend({
        model: BuildModel,

        initialize: function() {
            this.saveCurrentIds();
            this.on('request', this.saveCurrentIds);
        },

        parse: function(data) {
            return data.builds.map(function(build) {
                // append property 'just added' based on previously saved IDs
                var buildId = build[BuildModel.prototype.idAttribute];
                if (_.indexOf(this._previousIds, buildId, true) === -1) {
                    build.justAdded = true;
                }
                return build;
            }, this);
        },

        anyBuildSuccessful: function() {
            return this.some(function(model) {
                return model.isBuildSuccessful();
            });
        },

        anyBuildFailed: function() {
            return this.some(function(model) {
                return model.isBuildFailed();
            });
        },

        saveCurrentIds: function() {
            this._previousIds = _.pluck(this.models, 'id').sort();
        }

    }, {

        /**
         * Creates a new instance of builds collection for the given repository and branch
         *
         * @param repoId {number} id of repository
         * @param branchName {string} name of VCS branch
         * @param [sortBy] {string} how results should be sorted, leave empty for default ordering
         */
        forRepositoryBranch: function(repoId, branchName, sortBy) {
            return new this([], {
                url: bnUtils.getRestUrl('builds?repoId={0}&branchName={1}&sortBy={2}', repoId, encodeURIComponent(branchName), sortBy || '')
            });
        }
    });
});
